<?php
/**
 * If you're looking at this code to try and disable purchase verification -- well, I can't stop you.
 *
 * But, please consider that I am one developer working day and night to offer quality plugins
 * to the WordPress/WooCommerce community that solve real problems and save hard-working business
 * owners a lot of time and frustration. I am also not charging an arm and a leg like many other
 * plugins are.
 *
 * I currently get ~$25 from Envato for every $39 purchase (before taxes), which is barely enough
 * to cover my lunch and coffee in a day. But without this extra income I would not be able to
 * keep up with the necessary development and support.
 *
 * So please keep in mind, every version you use or share without buying is another nail in
 * the coffin that will eventually lead to this plugin being abandoned like so many before it.
 */

namespace Mewz\Framework\Services;

use Mewz\Framework\Plugin;

class Envato
{
	const URL = 'https://envato.mewz.dev/';

	/** @var Plugin */
	protected $plugin;

	/** @var int */
	protected $item_id;

	/** @var string */
	protected $token_name;

	/** @var string */
	protected $refresh_token_name;

	/** @var array */
	protected $cache = [];

	public function __construct(Plugin $plugin, $item_id)
	{
		$this->plugin = $plugin;
		$this->item_id = $item_id;

		$this->token_name = $plugin->prefix . '_envato_token';
		$this->refresh_token_name = $plugin->prefix . '_envato_refresh_token';
	}

	public function get_item_id()
	{
		return $this->item_id;
	}

	public function get_auth_url($return_url)
	{
		return self::URL . 'auth?return=' . urlencode($return_url);
	}

	public function authorized()
	{
		return strlen($this->get_refresh_token()) === 32;
	}

	public function validate()
	{
		if ($this->authorized() && $this->verify_purchase() === false) {
			$this->delete_tokens();
			return false;
		}

		return true;
	}

	public function get_plugin_info()
	{
		return $this->request('plugin_info', ['item_id' => $this->item_id], false);
	}

	public function verify_purchase()
	{
		$result = $this->request('verify_purchase', ['item_id' => $this->item_id]);

		if ($result instanceof \WP_Error) {
			return $this->is_envato_rejection($result) ? false : null;
		}

		return $result;
	}

	public function get_download_url()
	{
		$result = $this->request('download_url', ['item_id' => $this->item_id]);

		if ($result instanceof \WP_Error) {
			if ($this->is_envato_rejection($result)) {
				// if Envato refuses to give us a download url, assume the account no longer has access
				$this->delete_tokens();
			}

			return false;
		}

		return $result;
	}

	public function get_token()
	{
		if (!isset($this->cache['token'])) {
			$token = get_transient($this->token_name);

			if (!$token) {
				$new_token = $this->request_token();

				if (is_array($new_token)) {
					$this->save_tokens($new_token);
					$token = $new_token['access_token'];
				} else {
					return $new_token;
				}
			}

			$this->cache['token'] = $token;
		}

		return $this->cache['token'];
	}

	public function get_refresh_token()
	{
		global $wpdb;

		if (!isset($this->cache['refresh_token'])) {
			$this->cache['refresh_token'] = (string)$wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = '{$this->refresh_token_name}' LIMIT 1");
		}

		return $this->cache['refresh_token'];
	}

	public function save_tokens($params)
	{
		if (!empty($params['access_token']) && !empty($params['expires_in'])) {
			set_transient($this->token_name, $params['access_token'], (int)$params['expires_in'] - 600);
		}

		if (!empty($params['refresh_token'])) {
			update_option($this->refresh_token_name, $params['refresh_token'], false);
		}
	}

	public function delete_tokens()
	{
		global $wpdb;

		$wpdb->delete($wpdb->options, ['option_name' => $this->refresh_token_name]);

		delete_transient($this->token_name);

		$this->clear_cache();
	}

	public function request_token()
	{
		$refresh_token = $this->get_refresh_token();

		if (!$refresh_token) {
			return new \WP_Error('token_error', 'Not authorized');
		}

		return $this->request('access_token', ['refresh_token' => $refresh_token], false, false);
	}

	public function request($endpoint, array $params = [], $use_token = true, $use_cache = true, $token_retry = 3)
	{
		if ($use_cache && isset($this->cache[$endpoint])) {
			return $this->cache[$endpoint];
		}

		$headers = [];

		if ($use_token) {
			$token = $this->get_token();

			if (!$token) {
				$token = new \WP_Error('token_error', 'Failed to retrieve token');
			}

			if ($token instanceof \WP_Error) {
				$this->log_error($token);
				return $token;
			} else {
				$headers['Authorization'] = 'Bearer ' . $token;
			}
		}

		$request = [
			'headers' => $headers,
			'body' => $params,
		];

		$res = wp_remote_get(self::URL . $endpoint, $request);

		if (!$res instanceof \WP_Error) {
			$res['body'] = !empty($res['body']) ? json_decode($res['body'], true) : [];
			$res['response']['code'] = !empty($res['response']['code']) ? (int)$res['response']['code'] : 500;

			if (!empty($res['body']['error'])) {
				$code = !empty($res['body']['envato']) ? 'envato_error' : 'request_error';
				$res = new \WP_Error($code, $res['body']['error'], $res);
			}
			elseif ($res['response']['code'] >= 400) {
				$res = new \WP_Error('request_error', $res['response']['message'], $res);
			}
		}

		if ($res instanceof \WP_Error) {
			if ((int)$token_retry > 0 && strpos($res->get_error_message(), 'Invalid API token') !== false) {
				delete_transient($this->token_name);
				$this->clear_cache();

				return $this->request($endpoint, $params, $use_token, $use_cache, (int)$token_retry - 1);
			}

			$this->log_error($res, $request);
			$result = $res;
		} else {
			$result = $res['body'];
		}

		if ($use_cache) {
			$this->cache[$endpoint] = $result;
		}

		return $result;
	}

	public function is_envato_rejection($value)
	{
		return $value instanceof \WP_Error && $value->get_error_code() === 'envato_error' && $value->get_error_data()['response']['code'] !== 429;
	}

	public function clear_cache($key = null)
	{
		if ($key !== null) {
			unset($this->cache[$key]);
		} else {
			$this->cache = [];
		}
	}

	public function log_error(\WP_Error $error, $request = null)
	{
		$logger = wc_get_logger();
		if (!$logger) return;

		$data = $error->get_error_data();
		$message = $error->get_error_code();

		if (is_array($data) && !empty($data['response']['code'])) {
			$message .= ' - ' . $data['response']['code'];
		}

		$message .= ' - ' . $error->get_error_message();

		if (is_array($data) && !empty($data['http_response'])) {
			$message .= "\n URL: " . $data['http_response']->get_response_object()->url;
		}

		if ($request) {
			$message .= "\n Request: " . json_encode($request);
		}

		$logger->error($message, ['source' => $this->plugin->slug . '-envato']);
	}

	public function using_envato_plugin()
	{
		static $status;

		if ($status === null) {
			$status = false;

			if (defined('ENVATO_MARKET_VERSION') && function_exists('envato_market') && !get_option($this->plugin->prefix . '_envato_unlisted') && $settings = get_option(envato_market()->get_option_name())) {
				if (isset($settings['token']) && strlen($settings['token']) === 32) {
					$status = true;
				} elseif (!empty($settings['items'])) {
					foreach ($settings['items'] as $item) {
						if ($item['id'] == $this->item_id && $item['authorized'] === 'success' && strlen($item['token']) === 32) {
							$status = true;
							break;
						}
					}
				}
			}
		}

		return $status;
	}
}
