<?php
namespace Mewz\WCAS\Util;

use Mewz\WCAS\Aspects\Front\ProductLimits;

class Cart
{
	public static $cache = [];

	public static function get_cart_validation_items()
	{
		return apply_filters('mewz_wcas_cart_validation_items', WC()->cart->get_cart());
	}

	public static function get_cart_attribute_stock_limits($reverse_order = false)
	{
		$cart_items = self::get_cart_validation_items();
		$cart_limits = [];

		if ($reverse_order) {
			$cart_items = array_reverse($cart_items, true);
		}

		foreach ($cart_items as $cart_key => $cart_item) {
			$product = $cart_item['data'];

			if (!Products::is_valid_product($product) || Products::is_product_excluded($product)) {
				continue;
			}

			$variation = !empty($cart_item['variation']) ? $cart_item['variation'] : [];

			$stock_limits = Limits::get_stock_limits($product, $variation);
			if (!$stock_limits) continue;

			if (Components::is_sorted_tree($stock_limits)) {
				$deduct_limits = [];
				$component_tree = [];

				foreach ($stock_limits as $stock_id => $limit) {
					if (isset($cart_limits[$stock_id])) {
						$limit['stock_qty'] -= $cart_limits[$stock_id]['cart_qty'];
					}

					$deduct_limits[$stock_id] = $limit;

					if ($limit['comp']['use'] <= 0) {
						$component_tree[] = $stock_id;
					}
				}

				$comp_deductions = Components::calc_deductions($deduct_limits, $cart_item['quantity']);
			}

			foreach ($stock_limits as $stock_id => $limit) {
				if (isset($comp_deductions)) {
					$cart_quantity = $comp_deductions['deducted'][$stock_id] ?? 0;
				} else {
					$cart_quantity = $cart_item['quantity'] * $limit['multiplier'];
				}

				$cart_quantity = (float)apply_filters('mewz_wcas_cart_stock_item_quantity', $cart_quantity, $cart_item, $limit, $stock_limits);

				$cart_limits[$stock_id]['cart_qty'] = isset($cart_limits[$stock_id]['cart_qty'])
					? $cart_limits[$stock_id]['cart_qty'] + $cart_quantity
					: $cart_quantity;

				if (!isset($cart_limits[$stock_id]['stock_qty'])) {
					$cart_limits[$stock_id]['stock_qty'] = $limit['stock_qty'];
				}

				$cart_limits[$stock_id]['cart_items'][$cart_key] = $cart_item;

				if (isset($component_tree)) {
					if ($limit['comp']['use'] > 0) {
						$cart_limits[$stock_id]['component_tree'] = $component_tree;
					}
					elseif (isset($comp_deductions['dead_leaves'][$stock_id])) {
						$cart_limits[$stock_id]['component_used'] = true;
					}
				}
			}

			unset($comp_deductions, $component_tree);
		}

		$cart_limits = apply_filters('mewz_wcas_cart_stock_limits', $cart_limits, $cart_items);

		return $cart_limits;
	}

	public static function get_adding_over_shared_attribute_stock(\WC_Product $product, $quantity = 1, $variation = null, $cart_item_key = null)
	{
		if (!Products::is_valid_product($product) || Products::is_product_excluded($product)) {
			return false;
		}

		$quantity = apply_filters('mewz_wcas_cart_adding_over_shared_quantity', $quantity, $product, $variation, $cart_item_key);
		if ($quantity <= 0) return false;

		$adding_limits = Limits::get_stock_limits($product, $variation);
		$adding_limits = apply_filters('mewz_wcas_cart_adding_over_shared_adding_limits', $adding_limits, $product, $variation, $quantity, $cart_item_key);
		if (!$adding_limits) return false;

		$cart_limits = self::get_cart_attribute_stock_limits();
		$cart_limits = apply_filters('mewz_wcas_cart_adding_over_shared_cart_limits', $cart_limits, $adding_limits, $product, $variation, $quantity, $cart_item_key);
		if (!$cart_limits) return false;

		foreach ($adding_limits as $stock_id => &$adding) {
			if (isset($cart_limits[$stock_id])) {
				$adding['stock_qty'] -= $cart_limits[$stock_id]['cart_qty'];
				$adding['limit_qty'] = Matches::calc_limit_qty($adding['stock_qty'], $adding['multiplier']);
			}
		}

		$stock_qty = $product->get_stock_quantity();
		$available_qty = Limits::calc_limit_quantity($adding_limits);

		if ($quantity <= $available_qty) {
			return false;
		}

		$over_limit = [
			'stock_qty' => $stock_qty,
			'cart_qty' => $stock_qty - $available_qty,
		];

		return apply_filters('mewz_wcas_cart_adding_over_shared', $over_limit, $adding_limits, $cart_limits, $product, $variation, $quantity, $cart_item_key);
	}

	public static function get_over_shared_attribute_stock()
	{
		if (!isset(self::$cache['over_limits'])) {
			$cart_limits = self::get_cart_attribute_stock_limits();
			if (!$cart_limits) return false;

			$over_limits = [];

			foreach ($cart_limits as $stock_id => $limit) {
				if ($limit['cart_qty'] > $limit['stock_qty']) {
					if (!empty($limit['component_tree'])) {
						$limit['component_items'] = [];

						foreach ($limit['component_tree'] as $comp_id) {
							if (!empty($cart_limits[$comp_id]['component_used']) && !empty($cart_limits[$comp_id]['cart_items'])) {
								$limit['component_items'] += $cart_limits[$comp_id]['cart_items'];
							}
						}
					}

					$over_limits[$stock_id] = $limit;
				}
			}

			self::$cache['over_limits'] = apply_filters('mewz_wcas_cart_over_shared_stock_limits', $over_limits, $cart_limits);
		}

		return self::$cache['over_limits'];
	}

	/**
	 * Set selected variation data on cart item variation product objects so that correct
	 * attribute stock can be calculated for variations with catch-all attributes.
	 *
	 * @param array $cart_item
	 *
	 * @return array|false
	 */
	public static function set_item_product_variation_data($cart_item)
	{
		if (empty($cart_item['data'])) {
			return false;
		}

		/** @var \WC_Product $product */
		$product = $cart_item['data'];

		if (
			!empty($cart_item['variation_id'])
			&& !empty($cart_item['variation'])
			&& is_array($cart_item['variation'])
			&& $product instanceof \WC_Product_Variation
			&& Attributes::has_catchall($product->get_attributes())
		) {
			Products::set_prop($product, 'variation', $cart_item['variation']);
			return $cart_item['variation'];
		}
		elseif (Products::has_prop($product, 'variation')) {
			Products::set_prop($product, 'variation', null);
		}

		return false;
	}

	public static function multiply_cart_items($cart_items)
	{
		foreach ($cart_items as &$cart_item) {
			$product = $cart_item['data'];

			if (!$product instanceof \WC_Product) {
				continue;
			}

			if (($multiplier = Products::get_multiplier($product, 'product')) !== 1.00) {
				$cart_item['quantity'] = wc_stock_amount($cart_item['quantity'] * $multiplier);
			}
		}

		return $cart_items;
	}

	/**
	 * Check that the variation being added to cart has enough product-level stock (accounting for quantity already in the cart).
	 *
	 * Stock limited variations are filtered to have 'manage_stock' = true even if using product-level stock ('manage_stock' = parent)
	 * in order to avoid issues with parent stock validations when adding to cart. Specifically, parent stock is retrieved with `$variation->get_stock_quantity()`
	 * which, when variation stock is limited by attribute stock, does not reliably get the correct quantity of the product-level stock.
	 *
	 * Since native stock checking of variation product-level stock is effectively disabled for variations using attribute stock,
	 * we need to perform the checks ourselves.
	 *
	 * Error message is the default WooCommerce message from {@see WC_Cart::add_to_cart()}.
	 *
	 * @param \WC_Product_Variation $product
	 * @param float $quantity
	 *
	 * @return true|\WP_Error
	 */
	public static function validate_adding_parent_stock(\WC_Product_Variation $product, $quantity)
	{
		// all attribute stock limits should be bypassed as we only want to validate actual product stock
		ProductLimits::$enabled = false;

		if ($product->managing_stock() === 'parent' && $parent_id = $product->get_parent_id()) {
			$parent_cart_qty = 0;

			foreach (self::get_cart_validation_items() as $item) {
				if ($item['data']->get_stock_managed_by_id() === $parent_id) {
					$item_qty = $item['quantity'];

					if (($multiplier = Products::get_multiplier($item['data'], 'product')) !== 1.00) {
						$item_qty = wc_stock_amount($item_qty * $multiplier);
					}

					$parent_cart_qty += $item_qty;
				}
			}

			if ($parent_cart_qty > 0) {
				if (($multiplier = Products::get_multiplier($product, 'product')) !== 1.00) {
					$quantity = wc_stock_amount($quantity * $multiplier);
				}

				if (($parent_stock_qty = $product->get_stock_quantity()) < $parent_cart_qty + $quantity) {
					$message = sprintf(__('You cannot add that amount to the cart &mdash; we have %1$s in stock and you already have %2$s in your cart.', 'woocommerce'), wc_format_stock_quantity_for_display($parent_stock_qty, $product), wc_format_stock_quantity_for_display($parent_cart_qty, $product));

					$message = sprintf('<a href="%s" class="button wc-forward">%s</a> %s', esc_url(wc_get_cart_url()), esc_html__('View cart', 'woocommerce'), esc_html($message));

					$message = apply_filters('woocommerce_cart_product_not_enough_stock_already_in_cart_message', $message, $product, $parent_stock_qty, $parent_cart_qty);
				}
			}
		}

		ProductLimits::$enabled = true;

		return isset($message) ? new \WP_Error('insufficient_parent_stock', $message) : true;
	}

	/**
	 * Check that there is enough (shared) attribute stock for product being added to cart and products in the cart.
	 *
	 * @param \WC_Product $product
	 * @param float $quantity
	 * @param string $cart_item_key
	 *
	 * @return true|\WP_Error
	 */
	public static function validate_adding_shared_attribute_stock(\WC_Product $product, $quantity, $cart_item_key)
	{
		$over_limit = self::get_adding_over_shared_attribute_stock($product, $quantity, null, $cart_item_key);
		if (!$over_limit) return true;

		$message = sprintf(__('You cannot add that amount to the cart &mdash; we have %1$s in stock and you already have %2$s in your cart.', 'woocommerce'), wc_format_stock_quantity_for_display($over_limit['stock_qty'], $product), wc_format_stock_quantity_for_display($over_limit['cart_qty'], $product));

		$message = apply_filters('mewz_wcas_cart_add_stock_error', $message, $product, $quantity, $over_limit, $cart_item_key);

		$message = sprintf('<a href="%s" class="button wc-forward">%s</a> %s', esc_url(wc_get_cart_url()), esc_html__('View cart', 'woocommerce'), esc_html($message));

		$message = apply_filters('woocommerce_cart_product_not_enough_stock_already_in_cart_message', $message, $product, $over_limit['stock_qty'], $over_limit['cart_qty']);

		return new \WP_Error('insufficient_attribute_stock', $message);
	}

	/**
	 * Modified version of {@see \WC_Cart::check_cart_item_stock()} to check stock
	 * of cart items, but without taking into account stock of other items in the cart
	 * (we need to do this separately in a way that works with attribute stock).
	 *
	 * @return true|\WP_Error
	 */
	public static function check_individual_items_stock()
	{
		$current_session_order_id = isset(WC()->session->order_awaiting_payment) ? absint(WC()->session->order_awaiting_payment) : 0;

		foreach (WC()->cart->get_cart_contents() as $item) {
			/** @var \WC_Product $product */
			$product = $item['data'];

			if (!$product->is_in_stock()) {
				return new \WP_Error('out-of-stock', sprintf(__('Sorry, "%s" is not in stock. Please edit your cart and try again. We apologize for any inconvenience caused.', 'woocommerce'), $product->get_name()));
			}

			if (!$product->managing_stock() || $product->backorders_allowed()) {
				continue;
			}

			$held_stock = wc_get_held_stock_quantity($product, $current_session_order_id);

			/**
			 * Allows filter if product have enough stock to get added to the cart.
			 *
			 * @param bool $has_stock If have enough stock.
			 * @param \WC_Product $product Product instance.
			 * @param array $item Cart item values.
			 *
			 * @since 4.6.0
			 */
			if (apply_filters('woocommerce_cart_item_required_stock_is_not_enough', $product->get_stock_quantity() < ($held_stock + $item['quantity']), $product, $item)) {
				return new \WP_Error('out-of-stock', sprintf(__('Sorry, we do not have enough "%1$s" in stock to fulfill your order (%2$s available). We apologize for any inconvenience caused.', 'woocommerce'), $product->get_name(), wc_format_stock_quantity_for_display($product->get_stock_quantity() - $held_stock, $product)));
			}
		}

		return true;
	}

	/**
	 * Check that there is enough (shared) attribute stock for all items currently in the cart.
	 *
	 * @return true|\WP_Error
	 */
	public static function check_shared_attribute_stock()
	{
		$over_limits = self::get_over_shared_attribute_stock();
		if (!$over_limits) return true;

		$cart_items = [];

		foreach ($over_limits as $limit) {
			if (!empty($limit['component_items'])) {
				$cart_items += $limit['component_items'];
			} else {
				$cart_items += $limit['cart_items'];
			}
		}

		if (!$cart_items) return true;

		// we need backorder items to calculate stock limits for non-backorder items
		// but if all cart items are backorder items, then over-purchasing is allowed
		$all_allow_backorders = true;

		foreach ($cart_items as $cart_item) {
			if (!$cart_item['data']->backorders_allowed()) {
				$all_allow_backorders = false;
				break;
			}
		}

		if ($all_allow_backorders) return true;

		$product_names = [];
		$product_stocks = [];

		foreach ($cart_items as $cart_item) {
			/** @var \WC_Product $product */
			$product = $cart_item['data'];

			$product_names[] = $product->get_name();
			$product_stocks[] = $product->get_stock_quantity();
		}

		if ($product_names) {
			$product_names = implode(', ', array_unique($product_names));

			sort($product_stocks, SORT_NUMERIC);

			$product = current($cart_items)['data'];
			$product_stock = wc_format_stock_quantity_for_display($product_stocks[0], $product);

			if ($product_stocks[0] < end($product_stocks)) {
				$product_stock .= '–' . wc_format_stock_quantity_for_display(end($product_stocks), $product);
			}

			$message = sprintf(__('Sorry, we do not have enough "%1$s" in stock to fulfill your order (%2$s available). We apologize for any inconvenience caused.', 'woocommerce'), $product_names, $product_stock);

			$message = apply_filters('mewz_wcas_cart_stock_error', $message, current($over_limits), $over_limits);

			return new \WP_Error('insufficient_attribute_stock', $message);
		}

		return true;
	}

	/**
	 * Check that product being added to cart is in stock and has enough stock, but excluding
	 * quantity already in the cart or attribute stock on catch-all (any) variations will fail.
	 *
	 * Error messages are the default WooCommerce messages from {@see WC_Cart::add_to_cart()}.
	 *
	 * @param \WC_Product $product
	 * @param float $quantity
	 *
	 * @return true|\WP_Error
	 */
	public static function validate_adding_stock(\WC_Product $product, $quantity)
	{
		if (!$product->is_in_stock()) {
			$message = sprintf(__('You cannot add &quot;%s&quot; to the cart because the product is out of stock.', 'woocommerce'), $product->get_name());

			$message = apply_filters('woocommerce_cart_product_out_of_stock_message', $message, $product);
		}

		if (!$product->has_enough_stock($quantity)) {
			$message = sprintf(__('You cannot add that amount of &quot;%1$s&quot; to the cart because there is not enough stock (%2$s remaining).', 'woocommerce'), $product->get_name(), wc_format_stock_quantity_for_display($product->get_stock_quantity(), $product));

			$message = apply_filters('woocommerce_cart_product_not_enough_stock_message', $message, $product, $quantity);
		}

		if (isset($message)) {
			return new \WP_Error('insufficient_stock', $message);
		} else {
			return true;
		}
	}
}
