<?php
namespace Mewz\WCAS\Aspects\Front;

use Mewz\Framework\Base\Aspect;
use Mewz\Framework\Util\WooCommerce;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Limits;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Products;
use Mewz\WCAS\Util\Settings;

class VariableLimits extends Aspect
{
	public static $templates = [
		'single-product/add-to-cart/variable.php',
		'single-product/bundled-product-variable.php',
	];

	public $js_data = [];
	public $after_template;

	public function __hooks()
	{
		add_action('woocommerce_before_template_part', [$this, 'before_template_part'], 10, 4);
		add_action('wp_print_footer_scripts', [$this, 'output_js_data'], 0, 0);
	}

	public function output_js_data($ajax = false)
	{
		$js_data = $this->get_js_data();
		if (!$js_data) return;

		if (!$ajax) {
			$this->scripts->enqueue_js('@front/variable-stock');
			$this->scripts->export_data('variableData', $js_data);
		} else {
			// output js script and data within ajax rendered template so it can be executed dynamically
			$script_path = $this->assets->dir('dist/front/variable-stock.min.js');

			$this->view->render('front/variable-js-data-script', [
				'js_data' => $js_data,
				'inline_script' => file_get_contents($script_path),
			]);
		}
	}

	public function get_js_data()
	{
		$js_data = $this->js_data;

		if (!empty($js_data['data']) && !isset($js_data['images']) && $images = $this->get_image_data($js_data['data'])) {
			$js_data['images'] = $images;
		}

		if (!isset($js_data['settings'])) {
			$js_data['settings'] = [
				'no_stock_amount' => WooCommerce::no_stock_amount(),
				'low_stock_amount' => WooCommerce::low_stock_amount(),
				'hide_out_of_stock' => WooCommerce::hide_out_of_stock(),
				'product_stock_format' => Settings::product_stock_format(),
				'outofstock_variations' => Settings::outofstock_variations(),
				'unmatched_any_variations' => Settings::unmatched_any_variations(),
			];
		}

		// don't hate on the short keys - every byte counts!
		$js_data = apply_filters('mewz_wcas_variable_js_data', $js_data);

		if (!$js_data || empty($js_data['data'])) {
			return false;
		}

		return $js_data;
	}

	public function before_template_part($template, $path, $located, $args)
	{
		if (!in_array($template, self::$templates, true)) {
			return;
		}

		if (!empty($args['available_variations'])) {
			$variations = $args['available_variations'];
		}
		elseif (!empty($args['bundled_product_variations'])) {
			if ($args['bundled_item']->use_ajax_for_product_variations()) {
				return;
			}

			$variations = $args['bundled_product_variations'];
		}
		else return;

		if (isset($args['bundled_product'])) {
			$product = $args['bundled_product'];
		} else {
			/** @var \WC_Product_Variable $product */
			global $product;
		}

		if (!$product instanceof \WC_Product_Variable || isset($this->js_data['products'][$product->get_id()])) {
			return;
		}

		$hide_out_of_stock = WooCommerce::hide_out_of_stock() || in_array(Settings::outofstock_variations(), ['greyedout', 'hidden']);
		$display_stock = (bool)apply_filters('mewz_wcas_any_variations_display_stock', true, $product);

		$cache_key = 'variable_js_data_' . (int)$hide_out_of_stock . '_' . (int)$display_stock;
		$cache_tags = ['stock', 'match_rules', 'multipliers', 'product_' . $product->get_id()];

		$js_data = $this->cache->get($cache_key, $cache_tags);

		if ($js_data === null || $js_data === false) {
			$variations_data = [];
			$variations_attr = [];
			$oos_variations_count = 0;

			foreach ($variations as $variation) {
				if (empty($variation['is_in_stock'])) {
					$oos_variations_count++;
					continue;
				}

				if (!Attributes::has_catchall($variation['attributes'])) {
					continue;
				}

				$variation_id = $variation['variation_id'];
				$product_variation = wc_get_product($variation_id);

				if (!$product_variation || ($stock_status = $product_variation->get_stock_status()) === 'onbackorder') {
					continue;
				}

				$variation_data = [];

				if ($display_stock) {
					Products::incr_prop($product_variation, 'bypass_limits');
					$managing_stock = $product_variation->get_manage_stock();

					if ($managing_stock || $stock_status === 'instock') {
						Products::set_prop($product_variation, 'allow_backorders_override', true);

						$backorders = $product_variation->get_backorders();

						if ($backorders !== 'no') {
							$variation_data['b'] = $backorders;
						}

						if ($managing_stock) {
							$variation_data['m'] = $managing_stock;
							$variation_data['q'] = $product_variation->get_stock_quantity();
						}

						if (($multiplier = Products::get_multiplier([$product_variation, $product], 'product', null)) !== null) {
							$variation_data['px'] = $multiplier;
						}
					}

					Products::decr_prop($product_variation, 'bypass_limits');
				}

				if (($multiplier = Products::get_multiplier([$product_variation, $product], 'attribute', null)) !== null) {
					$variation_data['ax'] = $multiplier;
				}

				$variations_data[$variation_id] = $variation_data;
				$product_variations[$variation_id] = $product_variation;

				foreach ($variation['attributes'] as $attr => $term) {
					if ($term === '') {
						$variations_attr[$attr] = '';
					} elseif (!isset($variations_attr[$attr]) || $variations_attr[$attr] !== '') {
						$variations_attr[$attr][$term] = $term;
					}
				}
			}

			if (empty($variations_data) || empty($product_variations) || empty($variations_attr)) {
				$this->cache->set($cache_key, [], $cache_tags);
				return;
			}

			$variations_attr = Attributes::strip_attribute_prefix($variations_attr);

			$product_attr = Products::get_product_attributes($product, null, true) ?: [];
			$product_attr = Attributes::encode_keys($product_attr);

			foreach ($variations_attr as $attr => $terms) {
				if ($terms === '') {
					if (isset($product_attr[$attr]) && is_array($product_attr[$attr])) {
						$variations_attr[$attr] = $product_attr[$attr];
					}
				} elseif (is_array($terms)) {
					$variations_attr[$attr] = array_values($terms);
				}

				unset($product_attr[$attr]);
			}

			$all_attr = $variations_attr + $product_attr;

			if (!$all_attr) {
				$this->cache->set($cache_key, [], $cache_tags);
				return;
			}

			$match_data = Matches::get_any_match_data($product_variations, $all_attr, $product);

			if (!$match_data || empty($match_data['attributes'])) {
				$this->cache->set($cache_key, [], $cache_tags);
				return;
			}

			// trigger a warning if component tree is recursive
			if (isset($match_data['component_tree']) && $match_data['component_tree'] instanceof \WP_Error) {
				trigger_error($match_data['component_tree']->get_error_message(), E_USER_WARNING);
				unset($match_data['component_tree']);
			}

			if (!empty($match_data['matches']) && $hide_out_of_stock && $match_data['max_quantity'] <= 0 && count($variations_data) + $oos_variations_count === count($variations) && $this->compare_attribute_counts($match_data['attributes'], $all_attr)) {
				// all variations are out of stock
				$js_data = 0;
			} else {
				$variations_attr = Attributes::sluggify_attributes($variations_attr);
				$product_attr = Attributes::sluggify_attributes($product_attr);

				/**
				 * Hiding out of stock requires us to expand variations for ALL combinations of ALL attributes,
				 * because to hide an out of stock combination, we need to create variations for all combinations
				 * that are IN STOCK.
				 *
				 * The problem is that when it's a huge number of combinations it causes performance issues.
				 * The compromise here is to only expand all combinations under a max threshold when hiding out of stock,
				 * otherwise we use a filtered list of attributes for expanding combinations as normal.
				 */
				$filter_attributes = !$hide_out_of_stock;

				if (!$filter_attributes) {
					$total_combinations = $this->calculate_total_combinations($all_attr);
					$filter_attributes = $total_combinations >= apply_filters('mewz_wcas_any_variation_combination_threshold', 1000, $product);
				}

				if ($filter_attributes) {
					$variations_attr = $this->filter_attributes($variations_attr, $match_data['attributes']);
					$product_attr = $this->filter_attributes($product_attr, $match_data['attributes']);
				}

				$js_data = [
					'va' => $variations_attr,
					'pa' => $product_attr,
				];

				if (!empty($match_data['matches'])) {
					$js_data['md'] = $match_data['matches'];
					$js_data['vd'] = $variations_data;
				}

				if (!empty($match_data['term_multipliers'])) {
					$js_data['tm'] = $match_data['term_multipliers'];
				}

				if (!empty($match_data['component_tree'])) {
					$js_data['ct'] = $match_data['component_tree'];
				}
			}

			$this->cache->set($cache_key, $js_data, $cache_tags);
		}

		if ($js_data === 0) {
			// there are no variations in stock and "hide out of stock items" is enabled
			// so we need to short circuit the template output to show the product as out of stock
			$this->after_template = $template;
			add_action('woocommerce_after_template_part', [$this, 'after_template_part_override'], -100, 4);
			ob_start();
		}
		elseif ($js_data && is_array($js_data)) {
			if ($display_stock) {
				$js_data['sh'] = Limits::get_variable_stock_html($product);
			}

			$this->js_data['data'][$product->get_id()] = $js_data;

			if ($this->context->ajax_or_rest) {
				// the product form is being loaded via ajax, so we need to include variation data dynamically in the output
				$this->after_template = $template;
				add_action('woocommerce_after_template_part', [$this, 'after_template_part_jsdata'], -100, 4);
			}
		}
	}

	public function after_template_part_override($template, $path, $located, $args)
	{
		if ($template !== $this->after_template) {
			return;
		}

		$this->after_template = null;

		ob_end_clean();
		remove_action('woocommerce_after_template_part', [$this, 'after_template_part_override'], -100);

		if ($template === 'single-product/bundled-product-variable.php') {
			wc_get_template('single-product/bundled-product-unavailable.php', [
				'bundled_item' => $args['bundled_item'],
				'bundle' => $args['bundle'],
				'custom_product_data' => apply_filters('woocommerce_bundled_product_custom_data', [
					'is_unavailable' => 'yes',
					'is_out_of_stock' => 'yes',
					'is_required' => $args['bundled_item']->get_quantity('min', ['check_optional' => true]) > 0 ? 'yes' : 'no',
				], $args['bundled_item']),
			], false, WC_PB()->plugin_path() . '/templates/');
		} else {
			$args['available_variations'] = [];
			wc_get_template($template, $args);
		}
	}

	public function after_template_part_jsdata($template, $path, $located, $args)
	{
		if ($template !== $this->after_template) {
			return;
		}

		$this->after_template = null;
		remove_action('woocommerce_after_template_part', [$this, 'after_template_part_jsdata'], -100);

		$this->output_js_data(true);

		unset($this->js_data['data']);
	}

	public function compare_attribute_counts($attributes, $all_attr)
	{
		foreach ($all_attr as $key => $terms) {
			if ($terms && (empty($attributes[$key]) || count($attributes[$key]) < count($terms))) {
				return false;
		    }
		}

		return true;
	}

	public function filter_attributes($attributes, $filter)
	{
		$filtered = [];

		foreach ($attributes as $taxonomy => $terms) {
			if (empty($filter[$taxonomy])) continue;

			if (isset($filter[$taxonomy][''])) {
				$filtered[$taxonomy] = $terms;
				continue;
			}

			foreach ($terms as $term) {
				if (isset($filter[$taxonomy][$term])) {
					$filtered[$taxonomy][] = $term;
				}
			}
		}

		return $filtered;
	}

	public function calculate_total_combinations($attributes)
	{
	    $total = 1;

		foreach ($attributes as $terms) {
			if (!empty($terms)) {
				$total *= count($terms);
			}
		}

	    return $total;
	}

	public function get_image_data($product_data)
	{
		$images = [];

		foreach ($product_data as $data) {
			if (empty($data['md'])) continue;

			foreach ($data['md'] as $match) {
				if (empty($match['g'])) continue;

				$image_id = $match['g'];

				if (!isset($images[$image_id])) {
					$image = wc_get_product_attachment_props($image_id);

					if (!empty($image['url'])) {
						$images[$image_id] = $image;
					}
				}
			}
		}

		return $images;
	}
}
