<?php
namespace Mewz\WCAS\Aspects\Front;

use Mewz\Framework\Base\Aspect;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Cart;
use Mewz\WCAS\Util\Products;
use Mewz\WCAS\Util\Settings;

#TODO: Add "any" variation support for WooCommerce cart API.

class CartItems extends Aspect
{
	protected $cart_contents;
	protected $check_errors = 0;

	public function __hooks()
	{
		// validate stock limits *after* all other `add_to_cart` validations
		add_filter('woocommerce_add_cart_item', [$this, 'add_cart_item'], 0, 2);
		add_action('woocommerce_add_to_cart', [$this, 'added_to_cart'], -100, 6);
		add_action('woocommerce_bundled_add_to_cart', [$this, 'added_to_cart'], -100, 6);

		// override cart item quantities with multiplied product stock when calling WC_Cart::get_cart_item_quantities() during validate cart functions
		add_filter('woocommerce_get_cart_contents', [$this, 'check_cart_items_multiply']);

		// validate attribute stock in cart items
		add_action('woocommerce_check_cart_items', [$this, 'check_cart_items_before'], 0);
		add_action('woocommerce_check_cart_items', [$this, 'check_cart_items_after'], 2);

		// add data props to cart item product objects whenever cart items change
		add_filter('woocommerce_cart_contents_changed', [$this, 'cart_set_variation_data'], 5);
		add_filter('woocommerce_cart_contents_changed', [$this, 'cart_set_backorder_status'], 20);

		// add attribute stock backorder meta to order items
		add_action('woocommerce_checkout_create_order_line_item', [$this, 'checkout_create_order_line_item'], 5, 4);
	}

	public function add_cart_item($item, $key)
	{
		Cart::set_item_product_variation_data($item);

		return $item;
	}

	public function added_to_cart($cart_item_key, $product_id, $quantity = 1, $variation_id = 0, $variation = [], $cart_item_data = [])
	{
		$cart_item = WC()->cart->get_cart_item($cart_item_key);

		if (empty($cart_item['data']) || !$cart_item['data'] instanceof \WC_Product) {
			return;
		}

		$product = $cart_item['data'];

		if ($variation && $product instanceof \WC_Product_Variation && Attributes::has_catchall($product->get_attributes())) {
			Products::set_prop($product, 'variation', $variation);
		}

		do_action('mewz_wcas_before_validate_add_to_cart', $product, $cart_item_key, $quantity, $variation);

		$valid = true;

		if ($product->managing_stock() && !$product->backorders_allowed()) {
			$this->revert_cart_contents($cart_item_key, $quantity);

			$valid = $this->validate_adding_product($product, $quantity, $cart_item_key);
			$valid = apply_filters('mewz_wcas_validate_add_to_cart', $valid, $product, $cart_item_key, $quantity, $variation);

			$this->restore_cart_contents();
		}

		do_action('mewz_wcas_after_validate_add_to_cart', $valid, $product, $cart_item_key, $quantity, $variation);

		if ($valid instanceof \WP_Error) {
			$this->undo_added_item($cart_item_key, $quantity);
			throw new \Exception($valid->get_error_message());
		}
	}

	public function validate_adding_product(\WC_Product $product, $quantity, $cart_item_key)
	{
		if ($product instanceof \WC_Product_Variation) {
			// check stock quantities again for catch-all (any) variations now that the variation attributes are available to do a proper stock-limit check
			if (Products::has_prop($product, 'variation')) {
				$result = Cart::validate_adding_stock($product, $quantity);
				if ($result !== true) return $result;
			}

			// check that there's enough stock for variations using product-level stock
			$result = Cart::validate_adding_parent_stock($product, $quantity);
			if ($result !== true) return $result;
		}

		// check that enough attribute stock is available for this item and other items in the cart (shared stock quantity)
		$result = Cart::validate_adding_shared_attribute_stock($product, $quantity, $cart_item_key);
		if ($result !== true) return $result;

		return true;
	}

	public function check_cart_items_before()
	{
		// don't check attribute stock if checking cart items has been disabled
		if (!has_action('woocommerce_check_cart_items', [WC()->cart, 'check_cart_items'])) {
			return;
		}

		// first check product stock without any attribute stock applied
		ProductLimits::$enabled = false;

		// save current error count so we can check if there are any product stock errors in the next step
		$this->check_errors = wc_notice_count('error');
	}

	public function check_cart_items_after()
	{
		ProductLimits::$enabled = true;

		if (wc_notice_count('error') > $this->check_errors) {
			return;
		}

		if (
			// check stock of individual items, with attribute stock applied,
			// but without taking into account other items in the cart
			is_wp_error($result = Cart::check_individual_items_stock())

			// check that there's enough attribute stock between all items in the cart
			|| is_wp_error($result = Cart::check_shared_attribute_stock())
		) {
			wc_add_notice($result->get_error_message(), 'error');
		}
	}

	public function check_cart_items_multiply($cart_items)
	{
		if (!$cart_items) {
			return $cart_items;
		}

		$trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 10);
		$cart_item_quantities = false;
		$validating_cart = false;

		for ($i = 4; $i < 10; $i++) {
			if (!isset($trace[$i])) {
				break;
			}

			$function = $trace[$i]['function'];

			if (!$cart_item_quantities) {
				if ($function === 'get_cart_item_quantities') {
					$cart_item_quantities = true;
				}
				continue;
			}

			if (in_array($function, [
				'add_to_cart',
				'validate_add_to_cart',
				'validate_stock',
				'add_bundle_to_cart',
				'check_cart_item_stock',
				'check_cart_items',
				'validate_cart_item',
			])) {
				$validating_cart = true;
				break;
			}
		}

		if ($validating_cart) {
			$cart_items = Cart::multiply_cart_items($cart_items);
		}

		return $cart_items;
	}

	public function cart_set_variation_data($cart_contents)
	{
		Cart::$cache = [];

		foreach ($cart_contents as $item) {
			Cart::set_item_product_variation_data($item);
		}

		return $cart_contents;
	}

	public function cart_set_backorder_status($cart_contents)
	{
		if (!$cart_contents) {
			return $cart_contents;
		}

		foreach ($cart_contents as $item) {
			if (isset($item['data']) && $item['data'] instanceof \WC_Product && Products::has_prop($item['data'], 'backordered')) {
				Products::set_prop($item['data'], 'backordered', null);
				Products::set_prop($item['data'], 'stock_status', null);
			}
		}

		if (!$cart_limits = Cart::get_over_shared_attribute_stock()) {
			return $cart_contents;
		}

		foreach ($cart_limits as $cart_limit) {
			$backordered = $cart_limit['cart_qty'] - $cart_limit['stock_qty'];

			foreach ($cart_limit['cart_items'] as $item) {
				/** @var \WC_Product $product */
				$product = $item['data'];

				Products::incr_prop($product, 'backordered', $backordered);
				Products::set_prop($product, 'stock_status', 'onbackorder');
			}
		}

		return $cart_contents;
	}

	/**
	 * @param \WC_Order_Item_Product $order_item
	 * @param string $cart_item_key
	 * @param array $cart_item
	 * @param \WC_Order $order
	 */
	public function checkout_create_order_line_item($order_item, $cart_item_key, $cart_item, $order)
	{
		if (!$cart_item || !isset($cart_item['data']) || !$cart_item['data'] instanceof \WC_Product) {
			return;
		}

		$product = $cart_item['data'];
		$backordered = Products::get_prop($product, 'backordered', 0);

		if ($backordered > 0 && $product->backorders_require_notification()) {
			$meta_key = apply_filters('woocommerce_backordered_item_meta_name', __('Backordered', 'woocommerce'), $order_item);

			if (!$order_item->meta_exists($meta_key)) {
				$order_item->add_meta_data($meta_key, sprintf(__('%d (Shared)', 'woocommerce-attribute-stock'), $backordered));
			}
		}
	}

	protected function revert_cart_contents($cart_item_key, $quantity)
	{
		if ($this->cart_contents !== null) {
			return;
		}

		$cart = WC()->cart;

		if (!isset($cart->cart_contents[$cart_item_key])) {
			return;
		}

		$this->cart_contents = $cart->cart_contents;
		$cart->cart_contents[$cart_item_key]['quantity'] -= $quantity;

		if ($cart->cart_contents[$cart_item_key]['quantity'] <= 0) {
			unset($cart->cart_contents[$cart_item_key]);
		}
	}

	protected function restore_cart_contents()
	{
		if ($this->cart_contents === null) {
			return;
		}

		WC()->cart->cart_contents = $this->cart_contents;
		$this->cart_contents = null;
	}

	protected function undo_added_item($cart_item_key, $quantity)
	{
		$cart = WC()->cart;

		$cart_items = $this->get_all_related_cart_items($cart->cart_contents, $cart_item_key);
		if (!$cart_items) return;

		foreach ($cart_items as $item_key => $item) {
			$cart->set_quantity($item_key, $item['quantity'] - $quantity);
		}

		$cart->cart_contents = apply_filters('woocommerce_cart_contents_changed', $cart->cart_contents);
	}

	protected function get_all_related_cart_items($cart_contents, $cart_item_key)
	{
		if (empty($cart_contents[$cart_item_key])) {
			return [];
		}

		$items[$cart_item_key] = $cart_contents[$cart_item_key];

		if (!empty($cart_item['bundled_by']) && !empty($cart_contents[$cart_item['bundled_by']])) {
			$items[] = $bundle = $cart_contents[$cart_item['bundled_by']];

			if (!empty($bundle['bundled_items'])) {
				foreach ($bundle['bundled_items'] as $bundled_item_key) {
					if (!empty($cart_contents[$bundled_item_key])) {
						$items[] = $cart_contents[$bundled_item_key];
					}
				}
			}
		}

		return $items;
	}
}
