<?php
namespace Mewz\QueryBuilder;

/**
 * @todo stop coercing numeric strings to numbers
 * @todo add more aggregate query methods (https://laravel.com/docs/queries#aggregates)
 */

class Query
{
	/** @var array One or more tables to dispatch the query on */
	protected $tables = [];

	/** @var array Columns for SELECT clause */
	protected $select = [];

	/** @var bool Flag for DISTINCT in SELECT clause */
	protected $distinct = false;

	/** @var array Data key/value pairs for INSERT/UPDATE */
	protected $data = [];

	/** @var array Table JOIN clauses */
	protected $joins = [];

	/** @var array Conditions for WHERE clause */
	protected $where = [];

	/** @var array Columns to GROUP BY */
	protected $groupby = [];

	/** @var array Conditions for HAVING clause */
	protected $having = [];

	/** @var array Columns and orders to ORDER BY */
	protected $orderby = [];

	/** @var int Number of rows to retrieve */
	protected $limit;

	/** @var int Number of rows to offset results by */
	protected $offset;

	/** @var array Temporarily stores original clause data when overriding */
	protected $override = [];

	/* Clause Methods */

	/**
	 * Add a table to the query.
	 *
	 * @param string $name The table name, with or without prefix
	 * @param string|true $alias Optional table alias, or true to use $name as provided (without prefixing)
	 *
	 * @return $this
	 */
	public function table($name, $alias = null)
	{
		$this->tables[] = [$name, $alias];

		return $this;
	}

	/**
	 * Add a table to the query. Alias for {@see Query::table()}.
	 *
	 * @param string $name The table name, with or without prefix
	 * @param string|true $alias Optional table alias, or true to use $name as provided (without prefixing)
	 *
	 * @return $this
	 */
	public function from($name, $alias = null)
	{
		$this->tables[] = [$name, $alias];

		return $this;
	}

	public function select(...$columns)
	{
		foreach ($columns as $column) {
			$this->select[] = $column;
		}

		return $this;
	}

	public function distinct()
	{
		$this->distinct = true;

		return $this;
	}

	/**
	 * Eagerly add data to be used in an INSERT/UPDATE query.
	 *
	 * @param array $data [column => value] pairs
	 *
	 * @return $this
	 */
	public function data($data)
	{
		$this->data = $this->data ? array_merge($this->data, $data) : $data;

		return $this;
	}

	/**
	 * Add an INNER JOIN.
	 *
	 * @param string $name The table name, with or without prefix
	 * @param string|true $alias Optional table alias, or true to use $name as provided (without prefixing)
	 *
	 * @return $this
	 */
	public function join($name, $alias = null)
	{
		$this->joins[] = ['INNER JOIN', [$name, $alias]];

		return $this;
	}

	/**
	 * Add a LEFT JOIN.
	 *
	 * @param string $name The table name, with or without prefix
	 * @param string|true $alias Optional table alias, or true to use $name as provided (without prefixing)
	 *
	 * @return $this
	 */
	public function left_join($name, $alias = null)
	{
		$this->joins[] = ['LEFT JOIN', [$name, $alias]];

		return $this;
	}

	/**
	 * Add a RIGHT JOIN.
	 *
	 * @param string $name The table name, with or without prefix
	 * @param string|true $alias Optional table alias, or true to use $name as provided (without prefixing)
	 *
	 * @return $this
	 */
	public function right_join($name, $alias = null)
	{
		$this->joins[] = ['RIGHT JOIN', [$name, $alias]];

		return $this;
	}

	/**
	 * Add an ON clause to the last JOIN.
	 *
	 * @param string ...$condition Values passed directly to {@see DB::bind()}
	 *
	 * @return Query
	 */
	public function on(...$condition)
	{
		$this->joins[count($this->joins) - 1][2] = $condition;

		return $this;
	}

	public function groupby(...$columns)
	{
		foreach ($columns as $column) {
			$this->groupby[] = $column;
		}

		return $this;
	}

	public function orderby(...$columns)
	{
		foreach ($columns as $column) {
			$this->orderby[] = $column;
		}

		return $this;
	}

	public function asc(...$columns)
	{
		foreach ($columns as $column) {
			$this->orderby[] = $column . ' ASC';
		}

		return $this;
	}

	public function desc(...$columns)
	{
		foreach ($columns as $column) {
			$this->orderby[] = $column . ' DESC';
		}

		return $this;
	}

	public function newest($date_column)
	{
		$this->orderby[] = $date_column . ' DESC';

		return $this;
	}

	public function oldest($date_column)
	{
		$this->orderby[] = $date_column . ' ASC';

		return $this;
	}

	public function limit($limit, $offset = null)
	{
		$this->limit = (int)$limit;

		if ($offset !== null) {
			$this->offset = (int)$offset;
		}

		return $this;
	}

	public function offset($offset)
	{
		$this->offset = (int)$offset;

		return $this;
	}

	public function clear($clause)
	{
		$this->$clause = is_array($this->$clause) ? [] : null;

		return $this;
	}

	/* Query Methods */

	/**
	 * @see wpdb::get_results()
	 *
	 * @param int $limit
	 * @param int $offset
	 *
	 * @return array|null
	 */
	public function get($limit = null, $offset = null)
	{
		if ($limit !== null) {
			$this->override('limit', $limit);
		}

		if ($offset !== null) {
			$this->override('offset', $offset);
		}

		$sql = $this->build_select_query();

		return $sql ? DB::$wpdb->get_results($sql, ARRAY_A) : null;
	}

	/**
	 * @see wpdb::get_results()
	 * @see Query::page()
	 *
	 * @param int $page_number
	 * @param int $per_page
	 *
	 * @return array|null
	 */
	public function page($page_number = 1, $per_page = null)
	{
		$limit = $per_page === null ? (int)get_option('posts_per_page') : (int)$per_page;
		$offset = ((int)$page_number - 1) * $limit;

		return $this->get($limit, $offset);
	}

	/**
	 * @see wpdb::get_row()
	 *
	 * @return array|null
	 */
	public function row()
	{
		$this->override('limit', 1);

		$sql = $this->build_select_query();

		return $sql ? DB::$wpdb->get_row($sql, ARRAY_A) : null;
	}

	/**
	 * @see wpdb::get_col()
	 *
	 * @param string $column
	 *
	 * @return array
	 */
	public function col($column = null)
	{
		if ($column !== null) {
			$this->override('select', [$column]);
		}

		$sql = $this->build_select_query();

		return $sql ? DB::$wpdb->get_col($sql) : [];
	}

	/**
	 * @param string $key_column
	 * @param string $value_column
	 *
	 * @return array|null
	 */
	public function pairs($key_column, $value_column)
	{
		$this->override('select', [$key_column, $value_column]);

		$sql = $this->build_select_query();
		if (!$sql) return [];

		$rows = DB::$wpdb->get_results($sql, ARRAY_A);
		if (!$rows) return $rows;

		if ($dot = strpos($key_column, '.')) {
			$key_column = substr($key_column, $dot + 1);
		}

		if ($dot = strpos($value_column, '.')) {
			$value_column = substr($value_column, $dot + 1);
		}

		$pairs = [];

		foreach ($rows as $row) {
			$pairs[$row[$key_column]] = $row[$value_column];
		}

		return $pairs;
	}

	/**
	 * @see wpdb::get_var()
	 *
	 * @param string $column
	 *
	 * @return string|null
	 */
	public function val($column = null)
	{
		$this->override('limit', 1);

		if ($column !== null) {
			$this->override('select', [$column]);
		}

		$sql = $this->build_select_query();

		return $sql ? DB::$wpdb->get_var($sql) : null;
	}

	/**
	 * Gets the first row matching a condition.
	 *
	 * Shorthand for {@see where()} and {@see row()}, without changing the query data.
	 *
	 * @example DB::table('users')->find('user_id', 3);
	 *
	 * @param string $column
	 * @param mixed ...$value
	 *
	 * @return array|null
	 */
	public function find($column, ...$value)
	{
		$this->override('where');
		$this->where($column, ...$value);

		return $this->row();
	}

	/**
	 * Gets all rows matching a condition.
	 *
	 * Shorthand for {@see where()} and {@see get()}, without changing the query data.
	 *
	 * @example DB::table('posts')->find_all('post_author', [3, 4, 5]);
	 *
	 * @param string $column
	 * @param mixed ...$value
	 *
	 * @return array|null
	 */
	public function find_all($column, ...$value)
	{
		$this->override('where');
		$this->where($column, ...$value);

		return $this->get();
	}

	/**
	 * Counts the number of records for the current query.
	 *
	 * Query example: SELECT COUNT(*) FROM ...
	 *
	 * @param string $column The column to count (default '*')
	 *
	 * @return int|null The number of records, or null on failure
	 */
	public function count($column = '*')
	{
		$this->override('select', ["COUNT($column)"]);

		if ($sql = $this->build_select_query()) {
			$result = DB::$wpdb->get_var($sql);

			if ($result !== null) {
				return (int)$result;
			}
		}

		return null;
	}

	/**
	 * Efficiently checks if any records exist for the current query.
	 *
	 * Query example: SELECT EXISTS (SELECT * FROM ...)
	 *
	 * @return bool|null True/false if records exist, or null on failure
	 */
	public function exists()
	{
		if ($sql = $this->build_select_query()) {
			$sql = "SELECT EXISTS ($sql)";
			$result = DB::$wpdb->get_var($sql);

			if ($result !== null) {
				return (bool)$result;
			}
		}

		return null;
	}

	public function insert($data = [], $type = DB::INSERT)
	{
		if (!$this->tables || (!$data && !$this->data)) {
			return false;
		}

		if ($data) {
			$this->override('data');
			$this->data($data);
		}

		$sql = $this->build_insert_query($type);

		if ($this->override) {
			$this->override_reset();
		}

		$result = DB::$wpdb->query($sql);

		return $result && DB::$wpdb->insert_id ? DB::$wpdb->insert_id : $result;
	}

	public function replace($data = [])
	{
	    return $this->insert($data, DB::REPLACE);
	}

	public function update($data = [])
	{
		if (!$this->tables || !$this->where || (!$data && !$this->data)) {
			return false;
		}

		if ($data) {
			$this->override('data');
			$this->data($data);
		}

		$sql = $this->build_update_query();

		if ($this->override) {
			$this->override_reset();
		}

		return DB::$wpdb->query($sql);
	}

	public function delete()
	{
		$sql = $this->build_delete_query();

		return $sql ? DB::$wpdb->query($sql) : false;
	}

	public function alter($query)
	{
		if (!$this->tables) return false;

		$sql = 'ALTER TABLE ' . $this->get_main_table('name') . ' ' . $query;

		return DB::$wpdb->query($sql);
	}

	/**
	 * Truncates one or more tables, i.e. deletes all rows and resets auto-increment value.
	 *
	 * It goes without saying, but don't fuck around when using this method!
	 *
	 * @param bool $all_tables Whether to truncate all added tables or just the first (main) table.
	 *                         This is mainly for a bit of added protection against unintentional data loss.
	 *
	 * @return bool True on successful truncate, false on error
	 */
	public function truncate($all_tables = false)
	{
		if (!$this->tables) return false;

		if ($all_tables) {
			$sql = [];

			foreach ($this->tables as $table) {
				$table = $this->build_table($table, 'name');
				$sql[] = "TRUNCATE TABLE $table;";
			}

			$sql = implode("\n", $sql);
		} else {
			$sql = 'TRUNCATE TABLE ' . $this->get_main_table('name');
		}

		return DB::$wpdb->query($sql);
	}

	/**
	 * Drops one or more tables.
	 *
	 * It goes without saying, but don't fuck around when using this method!
	 *
	 * @param bool $if_exists Toggle 'IF EXISTS' after 'DROP TABLE'
	 * @param bool $all_tables Whether to drop all added tables or just the first (main) table.
	 *                         This is mainly for a bit of added protection against unintentional data loss.
	 *
	 * @return bool True on successful drop, false on error
	 */
	public function drop($if_exists = true, $all_tables = false)
	{
		if (!$this->tables) return false;

		$if_exists = $if_exists ? 'IF EXISTS ' : '';

		if ($all_tables) {
			$sql = [];

			foreach ($this->tables as $table) {
				$table = $this->build_table($table, 'name');
				$sql[] = "DROP TABLE $if_exists$table;";
			}

			$sql = implode("\n", $sql);
		} else {
			$sql = 'DROP TABLE ' . $if_exists . $this->get_main_table('name');
		}

		return DB::$wpdb->query($sql);
	}

	/* Utility Methods */

	public function inspect($prop = null)
	{
		if ($prop !== null) {
			return $this->$prop;
		} else {
			return get_object_vars($this);
		}
	}

	protected function override($prop, $value = null)
	{
		$this->override[$prop] = $this->$prop;

		if ($value !== null) {
			$this->$prop = $value;
		}
	}

	protected function override_reset()
	{
		foreach ($this->override as $prop => $value) {
			$this->$prop = $value;
		}

		$this->override = [];
	}

	protected function get_main_table($part = null)
	{
		if (!$this->tables) return false;

		return $this->build_table($this->tables[0], $part);
	}

	/* Build Methods */

	protected function build_tables($part = null)
	{
		$tables = [];

		foreach ($this->tables as $table) {
			$tables[] = $this->build_table($table, $part);
		}

		return implode(', ', $tables);
	}

	/**
	 * @param array $table [$name, $alias]
	 * @param null $part Optionally get only 'name' or 'alias'
	 *
	 * @return string
	 */
	protected function build_table(array $table, $part = null)
	{
		list($name, $alias) = $table;

		if ($alias !== true) {
			if ($part === 'alias') {
				if ($alias) {
					$name = $alias;
				}
			} else {
				if (isset(DB::$wpdb->$name) && is_string(DB::$wpdb->$name)) {
					$name = DB::$wpdb->$name;
				} else {
					$name = DB::prefix($name, true);
				}

				if ($part !== 'name' && $alias) {
					$name .= ' AS ' . $alias;
				}
			}
		}

		return $name;
	}

	protected function build_select()
	{
		$sql = 'SELECT ';

		if ($this->distinct) {
			$sql .= 'DISTINCT ';
		}

		if ($this->select) {
			return $sql . implode(', ', $this->select);
		} else {
			return $sql . '*';
		}
	}

	protected function build_joins()
	{
		$sql = [];

		foreach ($this->joins as $join) {
			$clause = $join[0] . ' ' . $this->build_table($join[1]);

			if (!empty($join[2])) {
				$clause .= ' ON ' . DB::bind(...$join[2]);
			}

			$sql[] = $clause;
		}

		return implode("\n", $sql);
	}

	protected function build_set()
	{
		$sql = [];

		foreach ($this->data as $key => $value) {
			$sql[] = '`' . $key . '` = ' . DB::prepare_value($value);
		}

		return 'SET ' . implode(', ', $sql);
	}

	protected function build_insert()
	{
		$data_rows = !is_array(current($this->data)) ? [$this->data] : array_values($this->data);
		$keys = $value_rows = [];

		foreach ($data_rows[0] as $key => $value) {
			$keys[] = '`' . $key . '`';
		}

		foreach ($data_rows as $row) {
			$values = [];

			foreach ($row as $value) {
				$values[] = DB::prepare_value($value);
			}

			$value_rows[] = '(' . implode(', ', $values) . ')';
		}

		return '(' . implode(', ', $keys) . ") VALUES\n" . implode(",\n", $value_rows) . ';';
	}

	protected function build_groupby()
	{
		return 'GROUP BY ' . implode(', ', $this->groupby);
	}

	protected function build_orderby()
	{
		return 'ORDER BY ' . implode(', ', $this->orderby);
	}

	protected function build_where()
	{
		return 'WHERE ' . $this->build_conditions($this->where);
	}

	protected function build_having()
	{
		return 'HAVING ' . $this->build_conditions($this->having);
	}

	protected function build_conditions($args)
	{
		$conditions = '';

		foreach ($args as $arg) {
			if ($conditions !== '') {
				$operator = isset($arg[2]) ? $arg[2] : 'AND';
				$conditions .= "\n $operator ";
			}

			$not = isset($arg[3]) && $arg[3] === 'NOT' ? 'NOT ' : '';

			if (!$arg[1]) {
				$conditions .= '(' . $not . $arg[0] . ')';
			} elseif (strpos($arg[0], '?') !== false) {
				$conditions .= '(' . $not . DB::bind($arg[0], ...$arg[1]) . ')';
			} elseif ($not) {
				$conditions .= DB::where(DB::NOT, $arg[0], ...$arg[1]);
			} else {
				$conditions .= DB::where($arg[0], ...$arg[1]);
			}
		}

		return $conditions;
	}

	protected function build_select_query()
	{
		if (!$this->tables) return false;

		$sql[] = $this->build_select();
		$sql[] = 'FROM ' . $this->build_tables();

		if ($this->joins) $sql[] = $this->build_joins();
		if ($this->where) $sql[] = $this->build_where();
		if ($this->groupby) $sql[] = $this->build_groupby();
		if ($this->having) $sql[] = $this->build_having();
		if ($this->orderby) $sql[] = $this->build_orderby();

		if ($this->limit !== null) $sql[] = 'LIMIT ' . $this->limit;
		if ($this->offset !== null) $sql[] = 'OFFSET ' . $this->offset;

		// reset overrides here once instead of doing it in all of the different select methods
		if ($this->override) {
			$this->override_reset();
		}

		return implode("\n", $sql);
	}

	protected function build_insert_query($type = DB::INSERT)
	{
		if (!$this->tables || !$this->data) {
			return false;
		}

		$table = $this->get_main_table('name');
		$insert = $this->build_insert();

		$type = $type === DB::REPLACE ? 'REPLACE' : 'INSERT';

		return "$type INTO $table $insert";
	}

	protected function build_update_query()
	{
		if (!$this->tables || !$this->data || !$this->where) {
			return false;
		}

		$tables = $this->build_tables();
		$set = $this->build_set();
		$where = $this->build_where();

		return "UPDATE $tables\n$set\n$where";
	}

	protected function build_delete_query()
	{
		if (!$this->tables) return false;

		$sql = 'DELETE';

		if ($this->joins || count($this->tables) > 1) {
			$sql .= ' ' . $this->build_tables('alias');
		}

		$sql .= ' FROM ' . $this->get_main_table();

		if ($this->joins) {
			$sql .= "\n" . $this->build_joins();
		}

		if ($this->where) {
			$sql .= "\n" . $this->build_where();
		}

		return $sql;
	}

	/* Condition Methods (WHERE/HAVING) */

	public function where($column, ...$value)
	{
		$this->where[] = [$column, $value];

		return $this;
	}

	public function having($column, ...$value)
	{
		$this->having[] = [$column, $value];

		return $this;
	}

	public function or_where($column, ...$value)
	{
		$this->where[] = [$column, $value, 'OR'];

		return $this;
	}

	public function or_having($column, ...$value)
	{
		$this->having[] = [$column, $value, 'OR'];

		return $this;
	}

	public function not($column, ...$value)
	{
		$this->where[] = [$column, $value, 'AND', 'NOT'];

		return $this;
	}

	public function or_not($column, ...$value)
	{
		$this->where[] = [$column, $value, 'OR', 'NOT'];

		return $this;
	}

	public function between($column, $from, $to)
	{
		$this->where[] = [$column, ['BETWEEN', $from, $to]];

		return $this;
	}

	public function not_between($column, $from, $to)
	{
		$this->where[] = [$column, ['NOT BETWEEN', $from, $to]];

		return $this;
	}

	public function like($column, $raw, ...$bindings)
	{
		array_unshift($bindings, 'LIKE', $raw);
		$this->where[] = [$column, $bindings];

		return $this;
	}

	public function not_like($column, $raw, ...$bindings)
	{
		array_unshift($bindings, 'NOT LIKE', $raw);
		$this->where[] = [$column, $bindings];

		return $this;
	}

	public function or_like($column, $raw, ...$bindings)
	{
		array_unshift($bindings, 'LIKE', $raw);
		$this->where[] = [$column, $bindings, 'OR'];

		return $this;
	}

	public function or_not_like($column, $raw, ...$bindings)
	{
		array_unshift($bindings, 'NOT LIKE', $raw);
		$this->where[] = [$column, $bindings, 'OR'];

		return $this;
	}

	public function is_null($column)
	{
		$this->where[] = ["$column IS NULL", []];

		return $this;
	}

	public function not_null($column)
	{
		$this->where[] = ["$column IS NOT NULL", []];

		return $this;
	}

	public function or_is_null($column)
	{
		$this->where[] = ["$column IS NULL", [], 'OR'];

		return $this;
	}

	public function or_not_null($column)
	{
		$this->where[] = ["$column IS NOT NULL", [], 'OR'];

		return $this;
	}

	/* Dump Methods */

	/**
	 * Return the full SQL statement.
	 *
	 * @param string $type One of DB::SELECT, DB::INSERT, DB::REPLACE, DB::UPDATE, or DB::DELETE
	 *
	 * @return string
	 */
	public function sql($type = DB::SELECT)
	{
		switch ($type) {
			case DB::SELECT:
				return $this->build_select_query();
			case DB::INSERT:
				return $this->build_insert_query();
			case DB::REPLACE:
				return $this->build_insert_query(DB::REPLACE);
			case DB::UPDATE:
				return $this->build_update_query();
			case DB::DELETE:
				return $this->build_delete_query();
			default:
				return '';
		}
	}

	/**
	 * @return string
	 */
	public function __toString()
	{
	    return $this->sql();
	}
}
