<?php
namespace Mewz\WCAS\Util;

use Mewz\Framework\Compatibility\Multilang;
use Mewz\Framework\Util\Number;
use Mewz\Framework\Util\WooCommerce;
use Mewz\QueryBuilder\DB;
use Mewz\WCAS\Models\AttributeStock;

class Matches
{
	const SETS_TABLE = 'wcas_match_sets';
	const ROWS_TABLE = 'wcas_match_rows';

	/**
	 * @param string $join
	 *
	 * @return \Mewz\QueryBuilder\Query
	 */
	public static function query($join = 'join')
	{
		return DB::table(self::SETS_TABLE, 's')
			->$join(self::ROWS_TABLE, 'r')->on('r.set_id = s.id')
			->distinct();
	}

	/**
	 * @param float $stock_qty
	 * @param float $multiplier
	 *
	 * @return int
	 */
	public static function get_limit_qty($stock_qty, $multiplier = 1)
	{
		if ($multiplier == 1 || $multiplier === '') {
			return (int)floor($stock_qty);
		} elseif ($multiplier <= 0) {
			return 0;
		} else {
			return (int)floor($stock_qty / $multiplier);
		}
	}

	/**
	 * Query stock for a specified attribute and optional term. Primarily for admin usage.
	 *
	 * @param int|string $attribute
	 * @param int $term_id
	 * @param string $context 'view' or 'edit'
	 * @param string $return 'object' or 'id'
	 *
	 * @return AttributeStock[]|int[]
	 */
	public static function query_stock($attribute, $term_id = null, $context = 'view', $return = 'object')
	{
		$attribute_id = Attributes::get_attribute_id($attribute);

		$cache_key = "query_stock_{$attribute_id}_{$term_id}_{$context}";
		$cache_tags = ['match_sets', 'stock', 'attribute_level'];

		$stock_ids = Mewz_WCAS()->cache->get($cache_key, $cache_tags);

		if (!is_array($stock_ids)) {
			$query = DB::table(DB::$wpdb->posts, 'p')
				->asc('p.post_title')
				->distinct();

			if ($term_id === null) {
				$query->left_join(DB::$wpdb->postmeta, 'pm_al')
					->on("pm_al.post_id = p.ID AND pm_al.meta_key = 'attribute_level'")
					->where('pm_al.meta_value', $attribute_id);
			} else {
				$query->left_join(self::SETS_TABLE, 's')->on('s.stock_id = p.ID')
					->left_join(self::ROWS_TABLE, 'r')->on('r.set_id = s.id')
					->where('r.attribute_id', $attribute_id)
					->where('r.term_id', $term_id);

				if ($context === 'edit') {
					$query->left_join(DB::$wpdb->postmeta, 'pm_al')
						->on("pm_al.post_id = p.ID AND pm_al.meta_key = 'attribute_level' AND pm_al.meta_value = $attribute_id")
						->is_null('pm_al.meta_value');
				}
			}

			if ($context === 'view') {
				$query->where('p.post_status', 'publish');
			} else {
				$query->not('p.post_status', ['trash', 'auto-draft']);
			}

			$stock_ids = $query->col('p.ID');

			// in view context, only return attribute term level stocks if any exist
			if ($term_id !== null && $context === 'view' && $stock_ids) {
				$attr_level_ids = DB::table(DB::$wpdb->postmeta)
					->where('post_id', $stock_ids)
					->where('meta_key', 'attribute_level')
					->where('meta_value', $attribute_id)
					->col('post_id');

				if (count($attr_level_ids) !== count($stock_ids)) {
					$stock_ids = array_values(array_diff($stock_ids, $attr_level_ids));
				}
			}

			Mewz_WCAS()->cache->set($cache_key, $stock_ids, $cache_tags);
		}

		if (!$stock_ids || !is_array($stock_ids)) {
			return [];
		}

		if ($return === 'object') {
			$stocks = [];

			foreach ($stock_ids as $stock_id) {
				$stock = AttributeStock::instance($stock_id, $context);

				if ($stock->exists()) {
					$stocks[] = $stock;
				}
			}

			return $stocks;
		} else {
			return $stock_ids;
		}
	}

	/**
	 * Finds all attribute stock items matching a product + attributes. Does not check for
	 * the "Limit product stock" setting.
	 *
	 * @param \WC_Product|int $product The product object or ID
	 * @param array $attributes Key/value pairs where key is an attribute id/name/taxonomy
	 *                          and value is a term id/slug or an array of term ids and/or slugs.
	 *                          Passed directly to {@see Matches::match_raw_stock()}.
	 * @param string $context 'view' or 'edit'
	 *
	 * @return array
	 */
	public static function match_product_stock($product, array $attributes, $context = 'view')
	{
		$raw_matches = self::match_raw_stock($attributes);
		$matches = [];

		if ($raw_matches) {
			$multiplier_override = Products::get_product_option('multiplier', $product);

			foreach ($raw_matches as $stock_id => $multipliers) {
				$stock = AttributeStock::instance($stock_id);

				if (!$stock->exists() || ($context === 'view' && !$stock->enabled()) || !self::validate_filters($stock, $product)) {
					continue;
				}

				if ($multiplier_override === '') {
					$multiplier = $stock->match_all() ? array_sum($multipliers) : $multipliers[0];
				} else {
					$multiplier = $stock->match_all() ? (float)$multiplier_override * count($multipliers) : (float)$multiplier_override;
				}

				$matches[] = [
					'stock_id' => $stock_id,
					'multiplier' => $multiplier,
				];
			}
		}

		/**
		 * IMPORTANT: The results from this filter are cached for performance reasons.
		 * This means the input ($product, $attributes) should always have the same output,
		 * or the cache must be invalidated accordingly.
		 *
		 * @see Limits::get_stock_limits()
		 */
		return apply_filters('mewz_wcas_product_stock_matches', $matches, $product, $attributes, $raw_matches);
	}

	/**
	 * Finds attribute stock items based solely on attributes. Does not check product filters
	 * or the "Limit product stock" setting.
	 *
	 * @param array $attributes Key/value pairs where key is an attribute id/name/taxonomy
	 *                          and value is a term id/slug or an array of term ids and/or slugs
	 *
	 * @return array Raw stock match results, before any validation or filtering
	 */
	public static function match_raw_stock(array $attributes)
	{
		global $wpdb;

		if (!$attributes) return [];

		$attributes = Attributes::get_attribute_id_sets($attributes);

		$cache_key = 'match_raw_stock_' . md5(json_encode($attributes));
		$cache_tags = ['match_sets'];

		$matches = Mewz_WCAS()->cache->get($cache_key, $cache_tags);

		if (!is_array($matches)) {
			$conditions = [];

			foreach ($attributes as $attribute_id => $term_ids) {
				foreach ($term_ids as $term_id) {
					$condition = 'r.attribute_id = ' . $attribute_id . ' AND r.term_id ';
					$condition .= $term_id ? 'IN (' . $term_id . ', 0)' : '= 0';

					$conditions[] = "($condition)";
				}
			}

			if (!$conditions) return [];

			$conditions = implode("\nOR ", $conditions);

			$rows_table = $wpdb->prefix . self::ROWS_TABLE;
			$sets_table = $wpdb->prefix . self::SETS_TABLE;

			// not using query builder here to save a little bit on performance as this method
			// will often be called many times in a row (e.g. when updating several orders at once)
			$query = "
				SELECT s.stock_id, IF(s.multiplier = '', 1, 0+s.multiplier) multiplier
				FROM {$sets_table} s
				JOIN {$rows_table} r ON r.set_id = s.id
				WHERE {$conditions}
				GROUP BY r.set_id
				HAVING COUNT(r.id) = (
					SELECT COUNT(*) FROM {$rows_table} r2
					WHERE r2.set_id = r.set_id
				)
				ORDER BY s.priority
			";

			$results = $wpdb->get_results($query);
			$matches = [];

			if ($results) {
				foreach ($results as $result) {
					$matches[(int)$result->stock_id][] = (float)$result->multiplier;
				}
			}

			Mewz_WCAS()->cache->set($cache_key, $matches, $cache_tags);
		}

		return $matches;
	}

	/**
	 * Checks if the configured attribute stock item filters match at least one of the provided products.
	 *
	 * @param AttributeStock $stock
	 * @param \WC_Product|\WC_Product[]|int|int[] $products
	 * @param bool $check_parent
	 * @param bool $bypass_multilang
	 *
	 * @return bool
	 */
	public static function validate_filters(AttributeStock $stock, $products, $check_parent = true, $bypass_multilang = true)
	{
		$bypass_multilang = $bypass_multilang && Multilang::active();

		foreach ((array)$products as $product) {
			if ($bypass_multilang) {
				$product_id = $product instanceof \WC_Product ? $product->get_id() : (int)$product;
				$product = Multilang::get_translated_object_id($product_id, 'post', 'product', 'default');
			}

			$valid = self::validate_filters_raw($stock, $product, $check_parent);
			$valid = apply_filters('mewz_wcas_validate_stock_filters', $valid, $stock, $product);

			if ($valid) return $valid;
		}

		return false;
	}

	/**
	 * @param AttributeStock $stock
	 * @param \WC_Product|int $product
	 * @param bool $check_parent
	 *
	 * @return bool
	 */
	public static function validate_filters_raw(AttributeStock $stock, $product, $check_parent = true)
	{
		static $check_bundles;

		if (!$product) return false;

		if ($product instanceof \WC_Product) {
			$product_ids[] = $product->get_id();

			if ($check_parent && $parent_id = $product->get_parent_id()) {
				$product_ids[] = $parent_id;
			}
		} else {
			if ($product instanceof \WC_Data_Store) {
				c($product);
				mewz_wcas_log(debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS), \WC_Log_Levels::INFO);
			}

			$product_ids[] = (int)$product;

			if ($check_parent && $parent_id = wp_get_post_parent_id($product_ids[0])) {
				$product_ids[] = $parent_id;
			}
		}

		$main_id = end($product_ids);

		if ($check_bundles || ($check_bundles === null && $check_bundles = function_exists('wc_pb_get_bundled_product_map'))) {
			$bundle_ids = wc_pb_get_bundled_product_map($main_id);

			if ($bundle_ids) {
				$product_ids = array_merge($product_ids, array_keys(array_flip($bundle_ids)));
			}
		}

		// if product or product ancestors are specifically excluded, immediately fail
		if (($excl_products = $stock->exclude_products()) && self::ids_in_array($product_ids, $excl_products)) {
			return false;
		}

		// if product or product ancestors are specifically included, immediately pass
		if (($incl_products = $stock->products()) && self::ids_in_array($product_ids, $incl_products)) {
			return true;
		}

		// if product type isn't valid, fail
		if ($valid_types = $stock->product_types()) {
			if ($product instanceof \WC_Product) {
				$product_type = $product->get_type();
			} else {
				$product_type = \WC_Product_Factory::get_product_type($main_id);
			}

			if ($product_type === 'variation') {
				$product_type = 'variable';
			}

			if (!in_array($product_type, $valid_types)) {
				return false;
			}
		}

		// check category filters
		$incl_cats = $stock->categories();
		$excl_cats = $stock->exclude_categories();

		if ($incl_cats || $excl_cats) {
			$product_cats = Products::get_all_product_category_ids($main_id);

			// if none of the product's categories are descendants of the included categories, fail
			if ($incl_cats && (!$product_cats || !self::ids_in_array($product_cats, $incl_cats))) {
				return false;
			}

			// if any of the product's categories are descendants of the excluded categories, fail
			if ($excl_cats && $product_cats && self::ids_in_array($product_cats, $excl_cats)) {
				return false;
			}
		}

		// if product filters are specified but don't match, and no other filters are specified, fail
		if ($incl_products && !$valid_types && !$stock->categories() && !$stock->exclude_categories()) {
			return false;
		}

		return true;
	}

	protected static function ids_in_array($ids, $array) {
		foreach ($ids as $id) {
			if (in_array($id, $array)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param int $stock_id
	 * @param string $context 'view' or 'edit'
	 *
	 * @return array [set_id] => [multiplier, rows[] => [attribute, term]]
	 */
	public static function get_sets($stock_id, $context = 'view')
	{
		$cache_key = "match_sets_{$stock_id}_{$context}_" . WooCommerce::get_cache_incr('woocommerce-attributes');
		$sets = Mewz_WCAS()->cache->get($cache_key, 'match_sets');

		if (!is_array($sets)) {
			$sets = [];

			$results = self::query()
				->select('r.set_id', 'r.id row_id', 's.multiplier', 'r.attribute_id', 'r.term_id')
				->asc('s.priority')
				->find_all('s.stock_id', $stock_id);

			if ($results) {
				$attributes = Attributes::get_attributes();

				foreach ($results as $result) {
					$set_id = (int)$result['set_id'];

					if (!isset($sets[$set_id])) {
						$multiplier = $result['multiplier'];

						if ($context === 'view') {
							$multiplier = $multiplier === '' ? 1 : (float)$multiplier;
						}

						$sets[$set_id] = [
							'multiplier' => $multiplier,
							'rows' => [],
						];
					}

					$attribute_id = (int)$result['attribute_id'];

					if (isset($attributes[$attribute_id])) {
						$sets[$set_id]['rows'][$attribute_id] = [
							'attribute' => $attribute_id,
							'term' => (int)$result['term_id'],
						];
					}
				}

				$row_sort_callback = static function($row1, $row2) use ($attributes) {
					return strnatcmp(
						$attributes[$row1['attribute']]['label'],
						$attributes[$row2['attribute']]['label']
					);
				};

				foreach ($sets as $set_id => &$set) {
					$row_count = count($set['rows']);

					if ($row_count === 0) {
						unset($sets[$set_id]);
					} elseif ($row_count > 1) {
						usort($set['rows'], $row_sort_callback);
					} else {
						$set['rows'] = array_values($set['rows']);
					}
				}
			}

			Mewz_WCAS()->cache->set($cache_key, $sets, 'match_sets');
		}

		if ($context === 'view') {
			$sets = apply_filters('mewz_wcas_get_match_sets', $sets, $stock_id);
		}

		return $sets;
	}

	/**
	 * @param int $stock_id
	 * @param array $sets Set format: [multiplier, priority, rows[] => [attribute, term]]
	 */
	public static function save_sets($stock_id, $sets)
	{
		do_action('mewz_wcas_match_sets_before_save', $stock_id, $sets);

		self::delete_sets($stock_id, null, false);

		$priority = 0;

		foreach ($sets as $set) {
			// make sure the set has at least one non-empty row
			if (!$set['rows'] || !array_filter(array_column($set['rows'], 'attribute'))) {
				continue;
			}

			if (isset($set['multiplier']) && $set['multiplier'] !== '') {
				$multiplier = is_string($set['multiplier']) ? (float)trim($set['multiplier']) : $set['multiplier'];
				$multiplier = $multiplier != 1 ? Number::period_decimal($multiplier) : '';
			} else {
				$multiplier = '';
			}

			$priority = isset($set['priority']) ? (int)$set['priority'] : $priority + 1;

			$set_id = DB::insert(self::SETS_TABLE, [
				'stock_id' => $stock_id,
				'multiplier' => $multiplier,
				'priority' => $priority,
			]);

			$attribute_rows = [];

			foreach ($set['rows'] as $row) {
				if (!empty($row['attribute']) && isset($row['term']) && $row['term'] !== '' && (int)$row['term'] >= 0) {
					$attribute_rows[] = [
						'set_id' => $set_id,
						'attribute_id' => (int)$row['attribute'],
						'term_id' => (int)$row['term'],
					];
				}
			}

			DB::insert(self::ROWS_TABLE, $attribute_rows);
		}

		Mewz_WCAS()->cache->invalidate('match_sets');

		do_action('mewz_wcas_match_sets_saved', $stock_id, $sets);
	}

	/**
	 * @param int $stock_id
	 * @param int|string $attribute
	 * @param int $term_id
	 * @param string|float $multiplier
	 *
	 * @return array [set_id, row_id]
	 */
	public static function add_set($stock_id, $attribute, $term_id = 0, $multiplier = '')
	{
		$attribute_id = Attributes::get_attribute_id($attribute);

		$max_priority = DB::table(self::SETS_TABLE)
			->where('stock_id', $stock_id)
			->val('MAX(priority) priority');

		$priority = (int)$max_priority + 1;
		$multiplier = (string)$multiplier === '' ? (string)$multiplier : Number::period_decimal((float)$multiplier);

		$set_id = DB::insert(self::SETS_TABLE, [
			'stock_id' => $stock_id,
			'multiplier' => $multiplier,
			'priority' => $priority,
		]);

		$row_id = DB::insert(self::ROWS_TABLE, [
			'set_id' => $set_id,
			'attribute_id' => $attribute_id,
			'term_id' => $term_id,
		]);

		Mewz_WCAS()->cache->invalidate('match_sets');

		return compact('set_id', 'row_id');
	}

	/**
	 * @param int $stock_id
	 * @param int $set_id
	 * @param bool $invalidate_cache
	 */
	public static function delete_sets($stock_id, $set_id = null, $invalidate_cache = true)
	{
		do_action('mewz_wcas_match_sets_before_delete', $stock_id, $set_id);

		$query = self::query('left_join')
			->table('r', true)
			->where('s.stock_id', $stock_id);

		if ($set_id !== null) {
			$query->where('s.id', $set_id);
		}

		$query->delete();

		// reset id auto increments to next highest values
		DB::table(self::SETS_TABLE)->alter('AUTO_INCREMENT = 1');
		DB::table(self::ROWS_TABLE)->alter('AUTO_INCREMENT = 1');

		if ($invalidate_cache) {
			Mewz_WCAS()->cache->invalidate('match_sets');
		}

		do_action('mewz_wcas_match_sets_deleted', $stock_id, $set_id);
	}

	/**
	 * Handles the removal of an attribute/term from all stock items.
	 * Related match groups will be removed, and empty stock items will be trashed.
	 *
	 * @param string|int $attribute
	 * @param int $term_id
	 *
	 * @return int[]|false
	 */
	public static function remove_attribute($attribute, $term_id = null)
	{
		$attribute_id = Attributes::get_attribute_id($attribute);

		// get affected stock ids
		$query = self::query()->where('r.attribute_id', $attribute_id);

		if ($term_id !== null) {
			$query->where('r.term_id', $term_id);
		}

		$stock_ids = $query->col('s.stock_id');
		if (!$stock_ids) return false;

		// delete all affected match sets and their rows
		$set_ids = $query->col('s.id');

		if ($set_ids) {
			self::query('left_join')
				->table('r', true)
				->where('s.id', $set_ids)
				->delete();

			DB::table(self::SETS_TABLE)->alter('AUTO_INCREMENT = 1');
			DB::table(self::ROWS_TABLE)->alter('AUTO_INCREMENT = 1');
		}

		Mewz_WCAS()->cache->invalidate('match_sets');

		// trash the stock items with no remaining match sets
		$trash_stock_ids = DB::table('posts', 'p')
			->left_join(self::SETS_TABLE, 's')->on('s.stock_id = p.ID')
			->where('p.ID', $stock_ids)
			->not('post_status', 'trash')
			->is_null('s.id')
			->col('p.ID');

		foreach ($trash_stock_ids as $stock_id) {
			(new AttributeStock($stock_id, 'object'))->trash();
		}

		Mewz_WCAS()->cache->invalidate('stock');

		if (!$term_id) {
			AttributeStock::delete_all_meta('attribute_level', $attribute_id);
			Mewz_WCAS()->cache->invalidate('attribute_level');
		}

		return $stock_ids;
	}

	public static function get_stock_attributes($post_status = 'all')
	{
		if ($post_status === 'all') {
			$post_status = ['publish', 'draft'];
		}

		$rows = self::query()
			->join(DB::$wpdb->posts, 'p')->on('p.ID = s.stock_id')
			->left_join(DB::$wpdb->postmeta, 'pm_al')->on("pm_al.post_id = p.ID AND pm_al.meta_key = 'attribute_level'")
			->where('p.post_status', $post_status)
			->select('r.attribute_id', 'r.term_id')
			->distinct()
			->get();

		$attr_level_ids = DB::table(DB::$wpdb->postmeta, 'pm')
			->join(DB::$wpdb->posts, 'p')->on('p.ID = pm.post_id')
			->where('p.post_type', 'mewz_attribute_stock')
			->where('p.post_status', $post_status)
			->where('pm.meta_key', 'attribute_level')
			->distinct()
			->col('pm.meta_value');

		$attributes = [];

		if ($rows) {
			foreach ($rows as $row) {
				$attributes[$row['attribute_id']][$row['term_id']] = (int)$row['term_id'];
			}
		}

		if ($attr_level_ids) {
			foreach ($attr_level_ids as $attr_id) {
				if (!isset($attributes[$attr_id])) {
					$attributes[$attr_id] = [];
				}
			}
		}

		return $attributes;
	}

	/**
	 * @param int $stock_id
	 * @param int|string $attribute
	 * @param int $term_id
	 *
	 * @return float
	 */
	public static function get_term_quantity_multiplier($stock_id, $attribute, $term_id)
	{
		$attribute_id = Attributes::get_attribute_id($attribute);

		$multiplier = self::query()
			->where('s.stock_id', $stock_id)
			->where('r.attribute_id', $attribute_id)
			->where('r.term_id', $term_id)
			->asc('s.priority')
			->val('s.multiplier');

		return (string)$multiplier === '' ? 1 : (float)$multiplier;
	}

	/**
	 * @param AttributeStock|int $stock
	 * @param int|string $attribute
	 * @param int $term_id
	 *
	 * @return float
	 */
	public static function get_term_available_quantity($stock, $attribute, $term_id)
	{
		if (!$stock instanceof AttributeStock) {
			$stock = AttributeStock::instance($stock);
		}

		$stock_qty = $stock->quantity();
		$multiplier = self::get_term_quantity_multiplier($stock->id(), $attribute, $term_id);

		return self::get_limit_qty($stock_qty, $multiplier);
	}

	/**
	 * @param AttributeStock|int $stock
	 * @param int|string $attribute
	 * @param int $term_id
	 *
	 * @return string
	 */
	public static function get_term_display_quantity($stock, $attribute, $term_id)
	{
		$available = self::get_term_available_quantity($stock, $attribute, $term_id);
		$quantity = $stock->quantity();

		$display = Number::local_format($available);

		if ($available != $quantity) {
			$display .= ' (' . Number::local_format($quantity) . ')';
		}

		return $display;
	}

	/**
	 * @param int[] $stock_ids
	 * @param int|string $attribute_id
	 * @param int $term_id
	 *
	 * @return string
	 */
	public static function get_quantity_range($stock_ids, $attribute_id, $term_id = null)
	{
		$quantities = [];

		foreach ($stock_ids as $stock_id) {
			$stock = AttributeStock::instance($stock_id);

			// show available stock for terms of attribute-level stock when displaying a quantity range
			if ($term_id && in_array($attribute_id, $stock->meta('attribute_level', false))) {
				$quantities[] = self::get_term_available_quantity($stock_id, $attribute_id, $term_id);
			} else {
				$quantities[] = $stock->quantity();
			}
		}

		$min_qty = min($quantities);
		$max_qty = max($quantities);

		if ($min_qty == $max_qty) {
			return Number::local_format($min_qty);
		} else {
			return Number::local_format($min_qty) . ' ... ' . Number::local_format($max_qty);
		}
	}

	/**
	 * @param \WC_Product|\WC_Product[] $product
	 * @param array $attributes
	 * @param bool $encode Whether to encode attribute names in result (needed for frontend usage)
	 *
	 * @return array|false
	 */
	public static function get_all_match_data($product, array $attributes, $encode = false)
	{
		if (empty($attributes)) return false;

		$attribute_ids = Attributes::get_attribute_id_sets($attributes);

		if (empty($attribute_ids)) return false;

		$set_ids = self::query()
			->join(DB::$wpdb->posts, 'p')->on('p.ID = s.stock_id')
			->where('p.post_type', AttributeStock::POST_TYPE)
			->where('p.post_status', 'publish')
			->where('r.attribute_id', array_keys($attribute_ids))
			->where('r.term_id', array_merge([0], ...$attribute_ids))
			->col('s.id');

		if (empty($set_ids)) return false;

		// get ALL match rows from found stock match sets
		$rows = self::query()
			->select('s.stock_id', 'r.set_id', 's.multiplier', 'r.attribute_id', 'r.term_id')
			->where('s.id', $set_ids)
			->asc('s.stock_id', 's.priority')
			->get();

		if (empty($rows)) return false;

		$matches = [];
		$attr_cache = [];

		$used_attributes = [];
		$max_quantity = 0;

		foreach ($rows as $row) {
			$stock_id = (int)$row['stock_id'];

			if (isset($valid_stock[$stock_id]) && $valid_stock[$stock_id] === false) {
				continue;
			}

			$stock = AttributeStock::instance($stock_id);

			if (!isset($valid_stock[$stock_id])) {
				if ($stock->exists() && $stock->enabled() && $stock->limit_products() && self::validate_filters($stock, $product)) {
					$valid_stock[$stock_id] = true;
				} else {
					$valid_stock[$stock_id] = false;
					continue;
				}
			}

			$set_id = (int)$row['set_id'];
			$attr_id = (int)$row['attribute_id'];

			if (!isset($attr_cache[$attr_id])) {
				$attr_name = Attributes::get_attribute_name((int)$row['attribute_id']);

				$attr_cache[$attr_id] = [
					$encode ? sanitize_title($attr_name) : $attr_name,
					'pa_' . $attr_name,
				];
			}

			list($attr_name, $taxonomy) = $attr_cache[$attr_id];

			if ($term_id = (int)$row['term_id']) {
				// we need the translated slug, and polylang doesn't auto translate get_term()
				if (Multilang::plugin() === 'polylang') {
					$term_id = Multilang::get_translated_object_id($term_id, 'term', $taxonomy);
				}

				$term_slug = Attributes::get_term_prop($term_id, $taxonomy, 'slug');
				if ($term_slug === false) continue;
			} else {
				$term_slug = ''; // any
			}

			if (!isset($matches[$stock_id])) {
				$matches[$stock_id]['q'] = $stock->quantity();

				if ($stock->match_all()) {
					$matches[$stock_id]['ma'] = true;
				}
			}

			if (!isset($matches[$stock_id]['s'][$set_id])) {
				$multiplier = $row['multiplier'] === '' ? 1 : (float)$row['multiplier'];

				if ($multiplier != 1) {
					$matches[$stock_id]['s'][$set_id]['x'] = $multiplier;
				}

				$available = self::get_limit_qty($stock->quantity(), $multiplier);

				if ($available > $max_quantity) {
					$max_quantity = $available;
				}
			}

			$matches[$stock_id]['s'][$set_id]['r'][$attr_name] = $term_slug;

			if (!isset($used_attributes[$taxonomy][$term_slug])) {
				$used_attributes[$taxonomy][$term_slug] = $term_slug;
			}
		}

		$matches = apply_filters('mewz_wcas_all_product_match_data', $matches, $product, $attributes, $max_quantity, $used_attributes);

		if ($matches) {
			// remove id keys from lists as they're not needed
			foreach ($matches as $stock_id => $match) {
				$matches[$stock_id]['s'] = array_values($match['s']);
			}

			$matches = array_values($matches);
		}

		if ($encode) {
			$used_attributes = Attributes::encode_keys($used_attributes);
		}

		return [
			'matches' => $matches,
			'attributes' => $used_attributes,
			'max_quantity' => $max_quantity,
		];
	}

	/**
	 * Generates a tax_query that matches attribute match rules to product attribute terms.
	 *
	 * @param array $match_sets
	 *
	 * @return array
	 */
	public static function get_sets_tax_query($match_sets)
	{
		$tax_query_list = [];

		foreach ($match_sets as $match_set) {
			$tax_queries = [];

			// build tax query
			foreach ($match_set['rows'] as $row) {
				$taxonomy = Attributes::get_attribute_name($row['attribute'], true);
				if (!$taxonomy) continue;

				$tax_query = ['taxonomy' => $taxonomy];

				if ($row['term']) {
					$tax_query['terms'] = $row['term'];
					$tax_query['operator'] = 'AND';
				} else {
					$tax_query['operator'] = 'EXISTS';
				}

				$tax_queries[] = $tax_query;
			}

			if (!$tax_queries) continue;

			if (!isset($tax_queries[1])) {
				$tax_queries = $tax_queries[0];
			}

			$tax_query_list[] = $tax_queries;
		}

		if ($tax_query_list) {
			if (!isset($tax_query_list[1])) {
				$tax_query_list = $tax_query_list[0];
			} else {
				$tax_query_list = ['relation' => 'OR'] + $tax_query_list;
			}
		}

		return $tax_query_list;
	}

	/**
	 * Queries a list of all products matching an attribute stock item.
	 *
	 * Important: This can be a fairly intensive operation. It should be used sparingly only
	 * when necessary, with appropriate use of the `$exclude` parameter.
	 *
	 * @param AttributeStock|int|array<AttributeStock|int> $stock Attribute stock item objects / IDs
	 * @param bool $query_variations Expand found variable products to matching variations
	 * @param int[] $exclude Product IDs to exclude
	 *
	 * @return array List of matching product IDs
	 */
	public static function query_matching_products($stock, $query_variations = false, $exclude = [])
	{
		$stock_ids = [];
		$product_ids = [];

		if (is_array($stock)) {
			foreach ($stock as $stock_item) {
				$matched_ids = self::query_matching_products($stock_item, false, $exclude);

				if ($matched_ids) {
					$product_ids[] = $matched_ids;
					$exclude = array_merge($exclude, $matched_ids);

					if ($query_variations) {
						$stock_ids[] = $stock_item instanceof AttributeStock ? $stock_item->id() : (int)$stock_item;
					}
				}
			}

			if ($product_ids) {
				$product_ids = array_merge(...$product_ids);
			}
		} else {
			if (!$stock instanceof AttributeStock) {
				$stock = AttributeStock::instance($stock);
			}

			$match_sets = self::get_sets($stock->id());
			if (!$match_sets) return [];

			$tax_query = self::get_sets_tax_query($match_sets);
			if (!$tax_query) return [];

			if ($query_variations) {
				$stock_ids[] = $stock->id();
			}

			$args = [
				'post_status' => ['publish', 'private', 'pending'],
				'tax_query' => [$tax_query],
				'return' => 'ids',
				'orderby' => ['ID' => 'ASC'],
				'limit' => -1,
				'update_post_meta_cache' => false,
				'update_post_term_cache' => false,
				'no_found_rows' => true,
				'suppress_filters' => true, // TODO: Is it safe to use this?
			];

			$include = $stock->products();
			if ($include) $args['include'] = $include;

			$exclude = array_keys(array_flip(array_merge($stock->exclude_products(), $exclude)));
			if ($exclude) $args['exclude'] = $exclude;

			$type = $stock->product_types();
			if ($type) $args['type'] = $type;

			$found_ids = wc_get_products($args);

			$include = array_flip($include);

			foreach ($found_ids as $found_id) {
				$found_id = (int)$found_id;

				if (!Products::is_product_excluded($found_id, false) && (isset($include[$found_id]) || self::validate_filters($stock, $found_id, false))) {
					$product_ids[] = $found_id;
				}
			}
		}

		if ($query_variations && $stock_ids && $product_ids) {
			$results = self::query_matching_variations($stock_ids, $product_ids);

			if ($results) {
				$product_ids = array_merge(array_diff($product_ids, $results['parent_ids']), $results['variation_ids']);
			}
		}

		return $product_ids;
	}

	/**
	 * @param int|int[] $stock_ids
	 * @param int|int[] $parent_ids
	 *
	 * @return array|false
	 */
	public static function query_matching_variations($stock_ids, $parent_ids)
	{
		global $wpdb;

		$parent_list = [];
		$pv_values = [];

		foreach ((array)$parent_ids as $parent_id) {
			$attributes = get_post_meta($parent_id, '_product_attributes', true);
			if (!$attributes) continue;

			$parent_list[] = $parent_id;

			foreach ($attributes as $taxonomy => $attr) {
				if (!empty($attr['is_taxonomy']) && !empty($attr['is_variation'])) {
					$pv_values[] = $parent_id;
					$pv_values[] = $taxonomy;
				}
			}
		}

		// no parent products have any variation attributes, so no variations to find
		if (!$pv_values) return false;

		$joins = [];
		$match_cond = [];

		foreach ((array)$stock_ids as $stock_id) {
			$match_sets = self::get_sets($stock_id);
			if (!$match_sets) continue;

			foreach ($match_sets as $match_set) {
				$group = [];

				foreach ($match_set['rows'] as $row) {
					$attr_id = (int)$row['attribute'];
					$taxonomy = Attributes::get_attribute_name($attr_id, true);

					$pv_alias = 'pv_' . $attr_id;
					$attr_alias = 'attr_' . $attr_id;

					if (!isset($joins[$attr_id])) {
						$joins[$pv_alias] = "LEFT JOIN __mewz_wcas_pv {$pv_alias} ON ({$pv_alias}.parent_id = p.post_parent AND {$pv_alias}.taxonomy = '{$taxonomy}')";
					}

					if ($row['term']) {
						$term_id = (int)$row['term'];
						$term_slug = Attributes::get_term_prop($term_id, $taxonomy, 'slug');

						$joins[$attr_alias] = "LEFT JOIN {$wpdb->postmeta} {$attr_alias} ON ({$attr_alias}.post_id = p.ID AND {$attr_alias}.meta_key = 'attribute_{$taxonomy}')";

						$match = "IF(
					        {$pv_alias}.parent_id IS NOT NULL,
					        {$attr_alias}.meta_value IS NULL OR {$attr_alias}.meta_value IN ('', '{$term_slug}'),
					        EXISTS(
								SELECT tr.term_taxonomy_id FROM {$wpdb->term_relationships} tr
								LEFT JOIN {$wpdb->term_taxonomy} tt ON (tt.term_taxonomy_id = tr.term_taxonomy_id)
								WHERE tr.object_id = p.post_parent AND tt.taxonomy = '{$taxonomy}' AND tt.term_id = {$term_id}
							)
					    )";
					} else {
						$match = "EXISTS(
							SELECT tr.term_taxonomy_id FROM {$wpdb->term_relationships} tr
							LEFT JOIN {$wpdb->term_taxonomy} tt ON (tt.term_taxonomy_id = tr.term_taxonomy_id)
							WHERE tr.object_id = p.post_parent AND tt.taxonomy = '{$taxonomy}'
						)";
					}

					$group[] = $match;
				}

				$match_cond[] = implode(' AND ', $group);
			}
		}

		if (!$joins || !$match_cond) {
			return false;
		}

		$joins = implode("\n", $joins);
		$parent_list = implode(',', $parent_list);
		$match_cond = '(' . implode(")\nOR (", $match_cond) . ')';
		$pv_insert = sprintf(rtrim(str_repeat("(%d,'%s'),", count($pv_values) / 2), ','), ...$pv_values);

		unset($pv_values); // allow memory to be freed

		$pv_create_query = "
			CREATE TEMPORARY TABLE __mewz_wcas_pv (
			    parent_id INT UNSIGNED NOT NULL,
			    taxonomy VARCHAR(32),
			    UNIQUE parent_taxonomy (parent_id, taxonomy)
			)
		";

		$pv_insert_query = "INSERT INTO __mewz_wcas_pv (parent_id, taxonomy) VALUES {$pv_insert}";
		$pv_drop_query = 'DROP TEMPORARY TABLE __mewz_wcas_pv';

		$main_query = "
			SELECT DISTINCT p.ID AS variation_id, p.post_parent AS parent_id
			FROM {$wpdb->posts} as p
			{$joins}
			WHERE p.post_type = 'product_variation'
			  AND p.post_status IN ('publish', 'private')
			  AND p.post_parent IN ({$parent_list})
			  AND (
			    {$match_cond}
			  )
			ORDER BY p.ID
		";

		if ($wpdb->query($pv_create_query) === false) {
			mewz_wcas_log('Failed to create temporary table in query_matching_variations(): ' . $wpdb->last_error, \WC_Log_Levels::ERROR);
			return false;
		}

		$wpdb->query($pv_insert_query);
		$results = $wpdb->get_results($main_query);
		$wpdb->query($pv_drop_query);

		if (!$results) return false;

		$variation_ids = [];
		$parent_ids = [];

		foreach ($results as $row) {
			if (!Products::is_product_excluded($row->variation_id, false)) {
				$variation_ids[] = (int)$row->variation_id;

				if (!isset($parent_ids[$row->parent_id])) {
					$parent_ids[$row->parent_id] = true;
				}
			}
		}

		if (!$parent_ids) return false;

		$parent_ids = array_keys($parent_ids);

		return compact('variation_ids', 'parent_ids');
	}
}
