<?php
namespace Mewz\WCAS\Plugin;

use Mewz\Framework\Plugin;
use Mewz\QueryBuilder\DB;
use Mewz\WCAS\Models\AttributeStock;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Settings;

class Installer extends Plugin\Installer
{
	public static function schema()
	{
		global $wpdb;

		$collate = $wpdb->get_charset_collate();

		return "
			CREATE TABLE " . $wpdb->prefix . Matches::SETS_TABLE . " (
			  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			  stock_id BIGINT UNSIGNED NOT NULL,
			  multiplier VARCHAR(100) NOT NULL,
			  priority INT(11) NOT NULL,
			  PRIMARY KEY  (id),
			  KEY stock_id (stock_id),
			  KEY multiplier (multiplier),
			  KEY priority (priority)
			) {$collate};

			CREATE TABLE " . $wpdb->prefix . Matches::ROWS_TABLE . " (
			  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			  set_id BIGINT UNSIGNED NOT NULL,
			  attribute_id BIGINT UNSIGNED NOT NULL,
			  term_id BIGINT UNSIGNED NOT NULL,
			  PRIMARY KEY  (id),
			  KEY set_id (set_id),
			  KEY attribute_id (attribute_id),
			  KEY term_id (term_id)
			) {$collate};
		";
	}

	public function init_data($operation)
	{
		if (MEWZ_WCAS_LITE) return;

		add_option('mewz_wcas_limit_product_stock', 'auto');
		add_option('mewz_wcas_allow_backorders', 'no');
		add_option('mewz_wcas_outofstock_variations', 'outofstock');
		add_option('mewz_wcas_unmatched_any_variations', 'no');
		add_option('mewz_wcas_sync_product_visibility', 'auto');
		add_option('mewz_wcas_trigger_product_stock_actions', 'yes');

		if ($operation !== 'update') {
			$this->maybe_optimize_tables();
			$this->maybe_sync_outofstock_products();
		}
	}

	public function migrations()
	{
		$db_version = $this->plugin->db_version;

		if (version_compare($db_version, '1.6.0', '<')) {
			$this->migrate_skus_to_metadata();
		}

		if (version_compare($db_version, '1.6.8', '<')) {
			$this->migrate_db_table_names();
		}

		if (version_compare($db_version, '1.9.0', '<')) {
			$this->maybe_optimize_tables();
			$this->maybe_sync_outofstock_products();
		}
	}

	public function uninstall()
	{
		if (defined('MEWZ_DELETE_DATA') && MEWZ_DELETE_DATA) {
			self::delete_options();
			self::delete_attribute_stock();
			self::delete_post_meta();
			self::drop_tables();
		}
	}

	public static function delete_options()
	{
		global $wpdb;

		DB::table($wpdb->options)->like('option_name', '?%', 'mewz_wcas_')->delete();
	}

	public static function delete_attribute_stock()
	{
		$stock_ids = AttributeStock::query(['post_status' => null], 'edit', 'id');

		foreach ($stock_ids as $stock_id) {
			(new AttributeStock($stock_id, 'object'))->delete(true);
		}
	}

	public static function delete_post_meta()
	{
		global $wpdb;

		DB::table($wpdb->postmeta)->like('meta_key', '?%', '_mewz_wcas_')->delete();
	}

	public static function drop_tables()
	{
		DB::table(Matches::SETS_TABLE)->drop();
		DB::table(Matches::ROWS_TABLE)->drop();
	}

	public function migrate_skus_to_metadata()
	{
		$query = DB::table('posts')->where('post_type', 'mewz_attribute_stock');
		$skus = $query->pairs('ID', 'post_excerpt');

		if (!$skus) return;

		foreach ($skus as $stock_id => $sku) {
			$sku = trim($sku);

			if ($sku !== '') {
				update_post_meta($stock_id, '_sku', $sku);
			}
		}

		$query->update(['post_excerpt' => '']);
	}

	public function migrate_db_table_names()
	{
		$pfx = DB::$wpdb->prefix;
		$old_tables = DB::$wpdb->get_col("SHOW TABLES LIKE '{$pfx}wc_mewz_wcas_%'");

		if (!$old_tables) return;

		$names = [
			$pfx . 'wc_mewz_wcas_match_sets' => $pfx . Matches::SETS_TABLE,
			$pfx . 'wc_mewz_wcas_match_rows' => $pfx . Matches::ROWS_TABLE,
		];

		foreach ($old_tables as $old_table) {
			if (!isset($names[$old_table])) {
				continue;
			}

			$new_table = $names[$old_table];

			if (DB::table($new_table)->count()) {
				DB::table($old_table)->drop();
			} else {
				DB::table($new_table)->drop();
				DB::table($old_table)->alter('RENAME TO ' . $new_table);
			}
		}
	}

	public function maybe_optimize_tables()
	{
		global $wpdb;

		if (get_option($this->plugin->prefix . '_skip_optimize_tables')) {
			return;
		}

		$postmeta_indexes = $wpdb->get_results("SHOW INDEX FROM {$wpdb->postmeta} WHERE Column_name = 'meta_value'");

		if (!$postmeta_indexes) {
			$suppressed = $wpdb->suppress_errors();
			$wpdb->query("ALTER TABLE {$wpdb->postmeta} ADD INDEX (meta_value(20))");
			$wpdb->suppress_errors($suppressed);
		}
	}

	public function maybe_sync_outofstock_products()
	{
		if (Settings::sync_product_visibility_bool()) {
			$this->plugin->tasks->add('update_all_outofstock');
		}
	}
}
