<?php
namespace Mewz\WCAS\Actions\Workers;

use Mewz\Framework\Base\Action;
use Mewz\WCAS\Models\AttributeStock;
use Mewz\WCAS\Util;

class StockChange extends Action
{
	public $affected_stock_ids = [];
	public $affected_product_ids = [];

	public function __hooks()
	{
		// trigger when attribute stock is saved/changed/trashed/untrashed/deleted
		add_action('mewz_attribute_stock_saved', [$this, 'attribute_stock_saved'], 10, 3);
		add_action('mewz_wcas_match_sets_saved', [$this, 'add_stock_ids']);
		add_action('transition_post_status', [$this, 'transition_post_status'], 10, 3);
		add_action('wp_trash_post', [$this, 'before_delete_post'], 10, 2);
		add_action('before_delete_post', [$this, 'before_delete_post'], 10, 2);

		// trigger stock changes in a background task
		add_action('shutdown', [$this, 'shutdown']);
		add_action('mewz_wcas_task_trigger_stock_changes', [$this, 'task_trigger_stock_changes']);
		add_action('mewz_wcas_stock_change', [$this, 'find_affected_product_stock']);

		// trigger product stock change actions in (another) background task
		if (Util\Settings::trigger_product_stock_actions() === 'yes') {
			add_action('mewz_wcas_product_stock_affected', [$this, 'product_stock_affected']);
			add_action('mewz_wcas_task_trigger_product_stock_changes', [$this, 'task_trigger_product_stock_changes']);
			add_action('mewz_wcas_trigger_product_stock_change', [$this, 'trigger_product_stock_change']);
		}
	}

	public function match_variations()
	{
	    return apply_filters('mewz_wcas_trigger_variation_stock_change_actions', true);
	}

	public function attribute_stock_saved($stock, $operation, $updated)
	{
		if (isset($updated['quantity']) || isset($updated['limit_products']) || isset($updated['match_all'])) {
			$this->add_stock_ids($stock->id());
		}
	}

	public function transition_post_status($new_status, $old_status, $post)
	{
		if ($post->post_type === 'mewz_attribute_stock' && !in_array($new_status, [$old_status, 'trash']) && in_array('publish', [$new_status, $old_status])) {
			$stock = AttributeStock::instance($post->ID);

			if (!$stock->limit_products()) {
				return;
			}

			$this->add_stock_ids($post->ID);
	    }
	}

	public function before_delete_post($post_id, $post = null)
	{
		$post_type = $post ? $post->post_type : get_post_type($post_id);

		if ($post_type !== 'mewz_attribute_stock') {
			return;
		}

		$stock = AttributeStock::instance($post_id);

		if (!$stock->exists() || !$stock->enabled() || !$stock->limit_products()) {
			return;
		}

		$stock_id = (int)$post_id;

		do_action('mewz_wcas_deleting_stock_change', $stock_id);

		// the stock item is being deleted, so we have to find affected products now
		if (Util\Settings::trigger_product_stock_actions() === 'yes') {
			$product_ids = Util\Matches::query_matching_products($stock_id, $this->match_variations());

			if ($product_ids) {
				$this->affected_product_ids = array_merge($this->affected_product_ids, $product_ids);
			}
		}
	}

	/**
	 * @param int|int[]|AttributeStock $stock
	 */
	public function add_stock_ids($stock)
	{
		if (!$stock) return;

		if ($stock instanceof AttributeStock) {
			$stock = $stock->id();
		}

		if (is_array($stock)) {
			$this->affected_stock_ids = array_merge($this->affected_stock_ids, $stock);
		} else {
			$this->affected_stock_ids[] = $stock;
		}
	}

	public function shutdown()
	{
		if ($this->affected_stock_ids) {
			$stock_ids = array_keys(array_flip($this->affected_stock_ids));
			sort($stock_ids, SORT_NUMERIC);

			do_action('mewz_wcas_stock_affected', $stock_ids);

			$this->tasks->add('trigger_stock_changes', ['stock_ids' => implode(',', $stock_ids)]);
		}

		if ($this->affected_product_ids) {
			$product_ids = array_keys(array_flip($this->affected_product_ids));
			sort($product_ids, SORT_NUMERIC);

			do_action('mewz_wcas_product_stock_affected', $product_ids);
		}
	}

	public function task_trigger_stock_changes($data)
	{
		if (empty($data['stock_ids'])) {
			return $this->tasks->kill(400);
		}

		$stock_ids = array_keys(array_flip(explode(',', $data['stock_ids'])));

		do_action('mewz_wcas_stock_change', $stock_ids);
	}

	public function find_affected_product_stock($stock_ids)
	{
		if (!has_action('mewz_wcas_product_stock_affected')) {
			return;
		}

		$product_ids = Util\Matches::query_matching_products($stock_ids, $this->match_variations());

		if ($product_ids) {
			do_action('mewz_wcas_product_stock_affected', $product_ids);
		}
	}

	public function product_stock_affected($product_ids)
	{
		$total = count($product_ids);
		$threshold = apply_filters('mewz_wcas_product_stock_update_multi_process_threshold', 50, $product_ids);

		if ($total >= $threshold) {
			$chunks = array_chunk($product_ids, ceil($total / 2));

			foreach ($chunks as $chunk) {
				$this->add_trigger_product_stock_changes_task($chunk);
			}
		} else {
			$this->add_trigger_product_stock_changes_task($product_ids);
		}
	}

	public function add_trigger_product_stock_changes_task($product_ids)
	{
		$this->tasks->add('trigger_product_stock_changes', ['product_ids' => implode(',', $product_ids)]);
	}

	public function task_trigger_product_stock_changes($data)
	{
		if (empty($data['product_ids'])) {
			return $this->tasks->kill(400);
		}

		if (!Util\Limits::product_limits_active()) {
			return $this->tasks->kill(501);
		}

		if (!has_action('mewz_wcas_trigger_product_stock_change')) {
			return $this->tasks->kill(501);
		}

		$product_ids = array_keys(array_flip(explode(',', $data['product_ids'])));

		if (!$product_ids) {
			return $this->tasks->kill(400);
		}

		// try to prevent other plugins getting confused and actually saving the overridden stock status
		add_action('woocommerce_before_product_object_save', [$this, 'before_product_object_save'], 0, 2);

		$match_variations = $this->match_variations();

		while ($product_id = array_shift($product_ids)) {
			$product = wc_get_product($product_id);

			if (!$product || ($match_variations && $product instanceof \WC_Product_Variable)) {
				continue;
			}

			do_action('mewz_wcas_trigger_product_stock_change', $product);

			if ($this->tasks->near_limits()) {
				$this->add_trigger_product_stock_changes_task($product_ids);
				break;
			}
		}
	}

	public function trigger_product_stock_change($product)
	{
		$type = $product instanceof \WC_Product_Variation ? 'variation' : 'product';

		do_action("woocommerce_{$type}_set_stock", $product);
		do_action("woocommerce_{$type}_set_stock_status", $product->get_id(), $product->get_stock_status(), $product);
	}

	public function before_product_object_save(\WC_Product $product)
	{
		if (apply_filters('mewz_wcas_product_stock_change_allow_status_save', false, $product)) {
			return;
		}

		$changes = $product->get_changes();

		if (!isset($changes['stock_status'])) {
			return;
		}

		$data = $product->get_data();

		$product->set_stock_status($data['stock_status']);

		if (count($changes) === 1) {
			$product->apply_changes();
		}
	}
}
