<?php
namespace Mewz\WCAS\Actions\Admin\Stock;

use Mewz\Framework\Base\Action;
use Mewz\Framework\Compatibility\Multilang;
use Mewz\Framework\Util\Admin;
use Mewz\QueryBuilder\DB;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Matches;

class StockListFilters extends Action
{
	public function __hooks()
	{
		add_filter('views_edit-mewz_attribute_stock', [$this, 'view_filters']);
		add_action('restrict_manage_posts', [$this, 'list_filters']);
	}

	public function __scripts()
	{
		$this->scripts->enqueue_js('wc-enhanced-select');
		$this->scripts->enqueue_css('woocommerce_admin_styles');
	}

	public function view_filters($views)
	{
		unset($views['mine'], $views['private'], $views['future']);

		$replace = [
			'publish' => __('Enabled', 'woocommerce-attribute-stock'),
			'draft' => __('Disabled', 'woocommerce-attribute-stock'),
		];

		foreach ($replace as $key => $label) {
			if (isset($views[$key])) {
				$views[$key] = preg_replace('/<a(.+?)>.+?<span/s', '<a$1>' . $label . ' <span', $views[$key]);
			}
		}

		return $views;
	}

	public function list_filters($post_type)
	{
	    if ($post_type !== 'mewz_attribute_stock') return;

	    ob_clean(); // removes all default post list filters

		$this->list_filter_stock();
		$this->list_filter_product();
		$this->list_filter_category();
		$this->list_filter_tag();
		$this->list_filter_attribute();
		$this->list_filter_term();
	}

	public function placeholder($text)
	{
	    return sprintf(_x('# %s', 'filter placeholder', 'woocommerce-attribute-stock'), $text);
	}

	public function list_filter_stock()
	{
		$this->view->render('admin/stock/list-filter', [
			'name' => 'stock',
			'placeholder' => $this->placeholder(__('Stock', 'woocommerce')),
			'options' => [
				'in-stock' => __('In stock', 'woocommerce'),
				'low-stock' => __('Low in stock', 'woocommerce'),
				'out-of-stock' => __('Out of stock', 'woocommerce'),
			],
		]);
	}

	public function list_filter_product()
	{
		$this->view->render('admin/stock/list-filter-product', [
			'name' => 'product_id',
			'placeholder' => $this->placeholder(__('Product', 'woocommerce')),
		]);
	}

	public function list_filter_category()
	{
		$post_status = !empty($_REQUEST['post_status']) && $_REQUEST['post_status'] !== 'all' ? $_REQUEST['post_status'] : ['publish', 'draft'];

		$results = DB::table('postmeta', 'pm')
			->left_join('posts', 'p')->on('p.ID = pm.post_id')
			->where('p.post_type', 'mewz_attribute_stock')
			->where('p.post_status', $post_status)
			->where('pm.meta_key', '_categories')
			->not('pm.meta_value', ['', 'a:0:{}'])
			->col('pm.meta_value');

		if ($results) {
			$category_ids = [];

			foreach ($results as $result) {
				$ids = unserialize($result);

				if ($ids && is_array($ids)) {
					foreach ($ids as $id) {
						$category_ids[$id] = true;
					}
				}
			}

			$options = Admin::get_category_options();

			if ($category_ids) {
				$options = array_intersect_key($options, $category_ids);
			}

			$this->view->render('admin/stock/list-filter', [
				'name' => 'category',
				'placeholder' => $this->placeholder(__('Category', 'woocommerce')),
				'options' => $options,
			]);
		}
	}

	public function list_filter_tag()
	{
		global $wpdb;

		$tags = $wpdb->get_results("
			SELECT DISTINCT t.name, t.slug
			FROM {$wpdb->terms} AS t
			LEFT JOIN {$wpdb->term_taxonomy} AS tt ON tt.term_id = t.term_id
			LEFT JOIN {$wpdb->term_relationships} AS tr ON tr.term_taxonomy_id = tt.term_taxonomy_id
			LEFT JOIN {$wpdb->posts} AS p ON p.ID = tr.object_id
			WHERE tt.taxonomy = 'product_tag'
			  AND p.post_type = 'mewz_attribute_stock'
			  AND p.post_status != 'auto-draft'
			ORDER BY t.name
		");

		if (!$tags) return;

		$options = [];

		foreach ($tags as $tag) {
			$options[$tag->slug] = $tag->name;
		}

		$this->view->render('admin/stock/list-filter', [
			'name' => 'product_tag',
			'placeholder' => $this->placeholder(__('Tag', 'woocommerce')),
			'options' => $options,
		]);
	}

	public function list_filter_attribute()
	{
		$attr_options = $this->get_attribute_options();
		$options = [];

		foreach ($attr_options as $attribute_id => $attribute) {
			$options[$attribute_id] = $attribute['label'];
		}

		$this->view->render('admin/stock/list-filter', [
			'name' => 'attribute',
			'placeholder' => $this->placeholder(__('Attribute', 'woocommerce')),
			'options' => $options,
		]);

		$this->scripts->export_js('@admin/stock-list', 'mewz_wcas_attribute_options', $attr_options);
	}

	public function list_filter_term()
	{
		$attr_options = $this->get_attribute_options();
		$attr_id = isset($_REQUEST['attribute']) && $_REQUEST['attribute'] !== '' ? (int)$_REQUEST['attribute'] : null;
		$options = [];

		if ($attr_id !== null && isset($attr_options[$attr_id])) {
			foreach ($attr_options[$attr_id]['terms'] as $term) {
				$options[$term[0]] = $term[1];
			}
		}

		$this->view->render('admin/stock/list-filter', [
			'name' => 'term',
			'placeholder' => $this->placeholder(__('Terms', 'woocommerce')),
			'options' => $options,
			'hidden' => !$options,
		]);
	}

	public function get_attribute_options()
	{
		static $options;

		if ($options === null) {
			$post_status = isset($_REQUEST['post_status']) ? $_REQUEST['post_status'] : 'all';
			$stock_attributes = Matches::get_stock_attributes($post_status);
			$attributes = Attributes::get_attributes();
			$options = [];

			Multilang::toggle_term_filters(false);

			foreach ($stock_attributes as $attr_id => $term_ids) {
				if (!isset($attributes[$attr_id])) {
					continue;
				}

				$terms = [];

				if ($term_ids) {
					$taxonomy = 'pa_' . $attributes[$attr_id]['name'];

					$term_labels = get_terms([
						'include' => array_filter($term_ids),
						'taxonomy' => $taxonomy,
						'fields' => 'id=>name',
						'orderby' => 'none',
						'hide_empty' => false,
						'update_term_meta_cache' => false,
					]);

					foreach ($term_labels as $term_id => $term_label) {
						$term_label = trim($term_label);

						if ($term_label === '') {
							$term_label = Attributes::get_term_prop($term_id, $taxonomy, 'slug') ?: (string)$term_id;
						} else {
							$term_label = html_entity_decode($term_label);
						}

						$terms[] = [$term_id, $term_label];
					}

					if (isset($term_ids[0])) {
						$terms[] = [0, '"' . __('Any', 'woocommerce-attribute-stock') . '"'];
					}
				}

				$options[$attr_id] = [
					'label' => $attributes[$attr_id]['label'],
					'terms' => $terms,
				];
			}

			Multilang::toggle_term_filters(true);

			uasort($options, static function($a, $b) {
				return strcmp($a['label'], $b['label']);
			});
		}

		return $options;
	}
}
