( function ( $, document ) {
	"use strict";

	var iconic_wpc = {
		$image_wrap: null,
		has_swatches: $( "[data-attribute-name]" ).length > 0,
		has_configurator: $( '.iconic-pc-images' ).length > 0,
		has_woothumbs: $( '.iconic-woothumbs-images' ).length > 0,
		$variations_form: $( 'form.variations_form' ),
		variation_selects: '.variations select',
		default_images: {},
		loaded_images: {},
		load_ids: {
			global: 0
		},

		/**
		 * On doc ready.
		 */
		on_ready: function () {
			iconic_wpc.set_elements();
			iconic_wpc.setup_default_images();
			iconic_wpc.setup_image_switcher();
			iconic_wpc.setup_swatches();
			iconic_wpc.setup_inventory();
			iconic_wpc.setup_woothumbs();
			iconic_wpc.setup_variation_swatches_photos();

			if ( iconic_wpc.has_swatches ) {
				iconic_wpc.reset_layers();
			}
		},

		/**
		 * Setup default images.
		 */
		setup_default_images: function() {
			iconic_wpc.$image_wrap.find( 'div' ).each( function( index, div ) {
				var $div = $( div ),
					id = $div.attr( 'class' ).match( /iconic-pc-image--([^\s]+)/ );

				if ( ! id ) {
					return;
				}

				var id_full_match = id[0];

				iconic_wpc.default_images[ '.' + id_full_match ] = $div.html();
			} );
		},

		/**
		 * Sanitise string. Similar to generating a slug.
		 *
		 * @param string str
		 * @param bool prefix
		 * @return {*}
		 */
		sanitise_str: function ( str, prefix ) {
			prefix = typeof prefix !== 'undefined' ? prefix : true;

			if ( str && str !== "" ) {
				if ( prefix ) {
					str = 'jckpc-' + str;
				}

				str = str.replace( '/', '' )
					.replace( / +/g, '-' )
					.replace( /[^a-zA-Z0-9-_]/g, '' )
					.replace( /(-)\1+/g, '-' )
					.replace( /(_)\1+/g, '_' );

				return str.toLowerCase();
			}

			return "";
		},

		/**
		 * Get layer ID.
		 *
		 * @param selectedAttName
		 * @return {string}
		 */
		get_layer_id: function ( selectedAttName ) {
			var prefix = '.iconic-pc-image--';

			if ( 0 === selectedAttName.indexOf( prefix ) ) {
				return selectedAttName;
			}

			selectedAttName = selectedAttName.replace( 'attribute_', '' );

			return prefix + selectedAttName;
		},

		/**
		 * Get default image.
		 *
		 * @param layer_id
		 */
		get_default_image: function ( layer_id ) {
			return iconic_wpc.default_images[layer_id];
		},

		/**
		 * Clear layer.
		 *
		 * @param layer_id
		 */
		clear_layer: function ( layer_id ) {
			layer_id = layer_id.indexOf( "." ) === 0 ? layer_id : iconic_wpc.get_layer_id( layer_id );

			var select_id = layer_id.replace( '.iconic-pc-image--jckpc-', '' ),
				$select = $( '[data-iconic_pc_layer_id="' + select_id + '"]' );

			if ( typeof $select === 'undefined' ) {
				return;
			}

			var select_value = $select.val(),
				$img = '' === select_value ? iconic_wpc.get_default_image( layer_id ) : '';

			$( layer_id ).html( $img );
		},

		/**
		 * Load image layer.
		 *
		 * @param selectedVal
		 * @param selectedAtt
		 */
		load_image_layer: function ( selectedVal, selectedAtt ) {
			var product_id = iconic_wpc.$image_wrap.data( 'product-id' );

			selectedAtt = selectedAtt.replace( 'attribute_', '' );

			var layer_id = iconic_wpc.get_layer_id( selectedAtt ),
				load_id = iconic_wpc.get_load_id( layer_id );

			if (typeof iconic_wpc.loaded_images[layer_id] === 'undefined' ) {
				iconic_wpc.loaded_images[layer_id] = {};
			}

			if ( typeof iconic_wpc.loaded_images[layer_id][selectedVal] !== 'undefined' ) {
				iconic_wpc.update_image_layer( layer_id, iconic_wpc.loaded_images[layer_id][selectedVal], load_id );
				return;
			}

			iconic_wpc.set_loading( true );
			iconic_wpc.load_json_images();
		},

		load_json_images: function () {
			var product_id = iconic_wpc.$image_wrap.data( 'product-id' ),
				selected_attributes = [],
				form_attributes = iconic_wpc.get_selected_attributes();

			for ( var attribute_name in form_attributes ) {
				selected_attributes[attribute_name.replace( 'attribute_', 'jckpc-' )] = form_attributes[attribute_name];
			}

			var conditional_images = iconic_wpc.get_conditional_images( selected_attributes );

			var json_layers = JSON.parse( $( '.iconic-pc-layers-' + product_id ).html() );

			if ( !json_layers ) {
				iconic_wpc.set_loading( false );
				return;
			}

			for ( var attribute in selected_attributes ) {
				var layer_id = iconic_wpc.get_layer_id( attribute ),
					image_html = '',
					load_id = iconic_wpc.get_load_id( layer_id ),
					selected_attr = selected_attributes[ attribute ];

				if ( typeof conditional_images[ attribute ] !== 'undefined' ) {
					var conditional_image = conditional_images[ attribute ][ selected_attr ];

					if ( typeof conditional_image !== 'undefined' ) {
						image_html = conditional_image.image_html;
					}
				}

				if ( '' === image_html ) {
					var default_layer = json_layers[attribute][selected_attr];
					image_html = typeof default_layer === 'undefined' ? '' : default_layer.image_html;
				}

				iconic_wpc.update_image_layer( layer_id, image_html, load_id );
			}

		},

		/**
		* Get conditional images
		*
		* @param selected_attributes
		*/
		get_conditional_images: function ( selected_attributes ) {
			var selected_images = [],
				product_id = iconic_wpc.$image_wrap.data( 'product-id' ),
				$conditionals = $( '.iconic-pc-layers-conditionals-' + product_id );

			if ( $conditionals.length <= 0 ) {
				return selected_images;
			}

			var conditionals = JSON.parse( $conditionals.html() );

			for ( var attribute in conditionals ) {
				for ( var rule_set in conditionals[attribute] ) {
					var rules_met = 0;

					for ( var current_rule in conditionals[attribute][rule_set].rules ) {
						var rule = conditionals[attribute][rule_set].rules[current_rule],
							selected_attr = selected_attributes[rule.attribute];

						if ( selected_attr ) {
							if ( 'is_equal_to' === rule.condition ) {
								if ( selected_attr === rule.value ) {
									rules_met++;
								}
							} else if ( 'is_not_equal_to' === rule.condition ) {
								if ( selected_attr !== rule.value ) {
									rules_met++;
								}
							}
						}
					}

					if ( rules_met === conditionals[attribute][rule_set].rules.length ) {
						for ( var current_value in conditionals[attribute][rule_set].value ) {
							var term = conditionals[attribute][rule_set].value[current_value];
							if ( 'undefined' === typeof selected_images[attribute] ) {
								selected_images[attribute] = [];
							}
							selected_images[attribute][term.attribute.replace( 'jckpc-', '' )] = term.value;
						}
					}
				}
			}

			return selected_images;
		},

		/**
		 * Set loading status.
		 *
		 * @param loading
		 */
		set_loading: function ( loading ) {
			if ( 'clear' === loading ) {
				iconic_wpc.load_ids.global = 0;
				return;
			}

			if ( loading ) {
				iconic_wpc.load_ids.global++;
				return;
			}

			if ( iconic_wpc.load_ids.global > 0 ) {
				iconic_wpc.load_ids.global--;
			}
		},

		/**
		 * Update image layer.
		 *
		 * @param layer_id
		 * @param image
		 * @param request_id
		 */
		update_image_layer: function ( layer_id, image, request_id ) {

			var current_load_id = iconic_wpc.get_load_id( layer_id );

			iconic_wpc.set_loading( false );

			if ( request_id !== current_load_id ) {
				return;
			}

			if ( image ) {
				$( layer_id ).html( image );
			} else {
				iconic_wpc.clear_layer( layer_id );
			}

			$( document.body ).trigger( 'iconic_pc_image_layer_updated' );
		},

		/**
		 * Generate dynamic image url.
		 *
		 * @param productId
		 * @return {*|string|string|boolean}
		 */
		generate_image_url: function ( productId ) {
			var url = jckpc.ajaxurl;

			url += url.indexOf( '?' ) < 0 ? '?' : '&';
			url += 'action=iconic_pc_generate_image';
			url += '&prodid=' + productId;
			url += '&' + iconic_wpc.get_selected_attributes( 'string' );

			return url;
			// http://iconic-plugins.local/wp-admin/admin-ajax.php?action=iconic_pc_generate_image&prodid=1177&attribute_strap=tan-leather&attribute_case=rose-gold&attribute_face=blue&attribute_pa_size=9&attribute_continents=anguilla
		},

		/**
		 * Get selected attributes.
		 *
		 * @param format
		 * @return {*}
		 */
		get_selected_attributes: function ( format ) {
			format = format || 'array';

			var selected = {},
				$selects = $( ".variations select[name^='attribute_']" );

			$selects.each( function ( index, select ) {
				var $select = $( select ),
					select_data = iconic_wpc.get_select_field_data( $select );

				if ( '' === select_data.value ) {
					var layer_id = iconic_wpc.get_layer_id( select_data.attribute ),
						$layer = $( layer_id );

					if ( $layer.length <= 0 ) {
						return;
					}

					var default_layer = $layer.data( 'iconic_pc_default_layer' );

					select_data.value = typeof default_layer !== 'undefined' ? default_layer : '';
				}

				selected[select_data.attribute_stripped] = select_data.value;
			} );

			if ( 'string' === format ) {
				return decodeURIComponent( $.param( selected, true ) );
			}

			return selected;
		},

		/**
		 * Setup image switcher.
		 */
		setup_image_switcher: function () {
			if ( iconic_wpc.has_configurator ) {

				$( document ).on( 'change', iconic_wpc.variation_selects, function () {

					var $selectField = $( this ),
						select_data = iconic_wpc.get_select_field_data( $selectField );

					iconic_wpc.increment_load_id( select_data.attribute );

					if ( select_data.value ) {
						iconic_wpc.load_image_layer( select_data.value, select_data.attribute );
					} else {
						iconic_wpc.clear_layer( select_data.attribute );
					}

					$( document.body ).trigger( 'iconic_pc_image_layer_updated' );
				} );

				$( document.body ).on( 'iconic_pc_image_layer_updated', function () {
					iconic_wpc.update_large_image();
				} );
			}
		},

		/**
		 * Increment load ID.
		 *
		 * @param attribute
		 */
		increment_load_id: function ( attribute ) {
			var $selects = $( '.variations select' );

			$selects.each( function ( index, select ) {
				var $select = $( select ),
					select_data = iconic_wpc.get_select_field_data( $select );

				var layer_id = iconic_wpc.get_layer_id( select_data.attribute );

				if ( !iconic_wpc.get_load_id( layer_id ) ) {
					iconic_wpc.load_ids[layer_id] = 0;
				}

				iconic_wpc.load_ids[layer_id]++;
			} );
		},

		/**
		 * Get current load ID.
		 *
		 * @param attribute
		 * @return {*}
		 */
		get_load_id: function ( attribute ) {
			var layer_id = iconic_wpc.get_layer_id( attribute );

			if ( typeof iconic_wpc.load_ids[layer_id] === 'undefined' ) {
				return false;
			}

			return iconic_wpc.load_ids[layer_id];
		},

		/**
		 * Setup swatches.
		 */
		setup_swatches: function () {
			if ( iconic_wpc.has_swatches && iconic_wpc.has_configurator ) {
				$( '.swatch-anchor' ).on( 'click', function () {
					var $selectedSwatchAnchor = $( this ),
						$variations_form = $selectedSwatchAnchor.closest( 'form' ),
						variationsMap = JSON.parse( $variations_form.attr( 'data-variations_map' ) ),
						select = $selectedSwatchAnchor.closest( '.select' ),
						swatch = $selectedSwatchAnchor.closest( '.select-option' ),
						selectedAttName = select.attr( 'data-attribute-name' ),
						selectedValHash = swatch.attr( 'data-value' ),
						selectedAtt = iconic_wpc.sanitise_str( selectedAttName ),
						selectedVal = variationsMap[selectedAttName][selectedValHash];

					if ( !swatch.hasClass( 'selected' ) ) {
						iconic_wpc.load_image_layer( selectedVal, selectedAtt );
					} else {
						iconic_wpc.clear_layer( selectedAtt );
					}

					$( document.body ).trigger( 'iconic_pc_image_layer_updated' );
				} );
			}
		},

		/**
		 * get swatch value.
		 *
		 * @param att_name
		 * @param att_val_hash
		 */
		get_swatch_value: function ( att_name, att_val_hash ) {
			var variationsMap = JSON.parse( iconic_wpc.$variations_form.attr( 'data-variations_map' ) );

			return variationsMap[att_name][att_val_hash];
		},

		/**
		 * Reset layers.
		 */
		reset_layers: function () {
			$( '#variations_clear' ).on( 'click', function () {
				$.each( iconic_wpc.default_images, function ( layer_id, image ) {
					iconic_wpc.$image_wrap.find( layer_id ).html( image );
				} );
			} );
		},

		/**
		 * Setup inventory.
		 */
		setup_inventory: function () {
			iconic_wpc.$variations_form.on( 'woocommerce_update_variation_values', function () {
				if ( typeof jckpc_inventory !== "undefined" ) {
					$.each( jckpc_inventory, function ( attribute_name, values ) {
						var $select = $( '#' + attribute_name );

						$.each( values, function ( attribute_option, inventory ) {
							var $option = $select.find( 'option[value="' + attribute_option + '"]' ),
								$va_picker = $( '.va-picker[data-attribute="' + attribute_name + '"][data-term="' + attribute_option + '"]' );

							if ( inventory !== "" && parseInt( inventory ) <= 0 ) {
								$option.attr( 'disabled', 'disabled' ).removeClass( 'enabled' );

								if ( $va_picker.length > 0 ) {
									$va_picker.hide();
								}
							}
						} );
					} );
				}
			} );

			// inventory for swatches plugin

			if ( iconic_wpc.has_swatches && iconic_wpc.has_configurator && typeof jckpc_inventory !== "undefined" ) {
				var $attribute_fields = $( '[data-attribute-name]' );

				$attribute_fields.each( function () {
					var $element = $( this ),
						attribute_name = $element.attr( 'data-attribute-name' ),
						$options = null;

					if ( $element.is( "select" ) ) {
						$options = $element.find( 'option' );

						$options.each( function ( index, option ) {

							var attribute_hash = $( option ).val();

							if ( attribute_hash !== "" ) {
								var attribute_value = iconic_wpc.get_swatch_value( attribute_name, attribute_hash ),
									attribute_name_formatted = attribute_name.replace( 'attribute_', '' ),
									attribute_value_formatted = attribute_value.replace( 'jckpc-', '' ),
									inventory = jckpc_inventory[attribute_name_formatted][attribute_value_formatted];

								if ( inventory !== "" && parseInt( inventory ) <= 0 ) {
									$( option ).remove();
								}
							}
						} );
					} else {
						$options = $element.find( '.select-option' );

						$options.each( function ( index, option ) {
							var attribute_hash = $( option ).attr( 'data-value' );

							if ( attribute_hash !== "" ) {
								var attribute_value = iconic_wpc.get_swatch_value( attribute_name, attribute_hash ),
									attribute_name_formatted = attribute_name.replace( 'attribute_', '' ),
									attribute_value_formatted = attribute_value.replace( 'jckpc-', '' ),
									inventory = jckpc_inventory[attribute_name_formatted][attribute_value_formatted];

								if ( inventory !== "" && parseInt( inventory ) <= 0 ) {
									$( option ).remove();
								}
							}
						} );
					}
				} );
			}
		},

		/**
		 * Set global elements.
		 */
		set_elements: function () {
			iconic_wpc.$image_wrap = $( '.iconic-pc-image-wrap' );
		},

		/**
		 * Setup WooThumbs.
		 */
		setup_woothumbs: function () {
			if ( !iconic_wpc.has_woothumbs ) {
				return;
			}

			/**
			 * Refresh WooThumbs after updated PC layers.
			 */
			$( document.body ).on( 'iconic_pc_image_layer_updated', function () {
				var $images = $( '.iconic-woothumbs-images.slick-initialized' );

				if ( $images.length <= 0 ) {
					return;
				}

				$images.slick( 'slickGoTo', 0 );
				$images.trigger( 'init_zoom' );
			} );

			/**
			 * Trigger change after WooThumbs reloads.
			 */
			$( '.iconic-woothumbs-all-images-wrap' ).on( 'iconic_woothumbs_images_loaded', function( event, product_object ) {
				var $first_attribute_select = product_object.variations_form.find( '.variations select' ).first();

				if ( $first_attribute_select.length <= 0 ) {
					return;
				}

				$first_attribute_select.change();
			} );
		},

		/**
		 * Update large image.
		 */
		update_large_image: function () {
			var product_id = parseInt( $( '.iconic-pc-image-wrap' ).data( 'product-id' ) ),
				url = iconic_wpc.generate_image_url( product_id ),
				$zoom_img = $( '.woocommerce-product-gallery__image--jckpc .zoomImg' );

			$( '.iconic-pc-image--background img' ).attr( 'data-large_image', url ).attr( 'data-src', url );

			// update default Woo zoom image.
			if ( $zoom_img.length > 0 ) {
				$zoom_img.attr( 'src', url );
			}
		},

		/**
		 * Get select field data.
		 *
		 * @param $select
		 * @return {{attribute: null, value: null}}
		 */
		get_select_field_data: function ( $select ) {
			var data = {
				'attribute_stripped': null,
				'attribute': null,
				'value': null
			};

			if ( iconic_wpc.has_swatches ) {
				var variationsMap = JSON.parse( iconic_wpc.$variations_form.attr( 'data-variations_map' ) ),
					selectedAttName = $select.attr( 'data-attribute-name' ),
					selectedValHash = $select.val();

				data.attribute = iconic_wpc.sanitise_str( selectedAttName );
				data.attribute_stripped = data.attribute.replace( 'jckpc-', '' );
				data.value = variationsMap[selectedAttName][selectedValHash];

				return data;
			}

			data.attribute = iconic_wpc.sanitise_str( $select.attr( 'name' ) );
			data.attribute_stripped = data.attribute.replace( 'jckpc-', '' );
			data.value = $select.val();

			return data;
		},

		/**
		 * Compatibility with WooCommerce Variations Swatches and Photos Plugin.
		 */
		setup_variation_swatches_photos: function () {
			if ( jQuery( '.variations_form.swatches' ).length ) {
				var variationAttributesInputs = '.variations_form.swatches .variation_form_section input:hidden';

				iconic_wpc.variation_selects += ', ' + variationAttributesInputs;
				iconic_wpc.has_swatches = false;

				$( variationAttributesInputs ).each( function () {
					var $input = $( this );
					$input.attr( 'data-attribute-name', $input.attr( 'name' ) );

					$input.parent().find( '.select-option a' ).click( function () {
						setTimeout( function () {
							$input.trigger( 'change' );
						}, 200 );
					} );
				} );
			}
		}
	};

	$( document ).ready( iconic_wpc.on_ready() );
}( jQuery, document ) );