/**
 * WordPress dependencies
 */
import { MediaUploadCheck, MediaUpload } from '@wordpress/block-editor';
import { ToolbarButton, Dropdown, NavigableMenu, MenuItem } from '@wordpress/components';
import { __, sprintf } from '@wordpress/i18n';
import { linkOff, image as imageIcon } from '@wordpress/icons';
import { VideoControlProps } from '../../types';
/**
 * Internal dependencies
 */
import styles from './style.module.scss';
import { VideoPosterCardProsp } from './types';
import type React from 'react';

const VIDEO_POSTER_ALLOWED_MEDIA_TYPES = [ 'image' ];

/**
 * Simple component that renders info about video poster.
 *
 * @param {VideoPosterCardProsp} props - Component props.
 * @returns {React.ReactElement}         VideoPosterCard component
 */
function VideoPosterCard( { poster }: VideoPosterCardProsp ) {
	const notes = __(
		'No custom Poster image selected. You can upload or select an image from your media library to override the default video image.',
		'jetpack-videopress-pkg'
	);

	const overridingNotes = __(
		'You are currently overriding the default Poster image. Remove it if you want to use the default image generated by VideoPress.',
		'jetpack-videopress-pkg'
	);

	return <div className={ styles.current_media }>{ poster ? overridingNotes : notes }</div>;
}

/**
 * Poster image control react component.
 *
 * @param {VideoControlProps} props - Component props.
 * @returns {React.ReactElement}      PosterImageBlockControl block control
 */
export default function PosterImageBlockControl( {
	attributes,
	setAttributes,
	clientId,
}: VideoControlProps ) {
	const { poster } = attributes;
	const onSelectPoster = image => {
		setAttributes( { poster: image.url } );
	};

	const onRemovePoster = () => {
		setAttributes( { poster: '' } );
	};

	const selectPosterLabel = __( 'Select Poster Image', 'jetpack-videopress-pkg' );
	const replacePosterLabel = __( 'Replace Poster Image', 'jetpack-videopress-pkg' );

	return (
		<Dropdown
			contentClassName={ styles.dropdown_content }
			renderToggle={ ( { isOpen, onToggle } ) => (
				<ToolbarButton
					label={ __( 'Poster image', 'jetpack-videopress-pkg' ) }
					showTooltip
					aria-expanded={ isOpen }
					aria-haspopup="true"
					onClick={ onToggle }
					icon={ imageIcon }
				/>
			) }
			renderContent={ ( { onClose } ) => {
				const videoPosterDescription = `video-block__poster-image-description-${ clientId }`;
				return (
					<>
						<NavigableMenu className="block-editor-media-replace-flow__media-upload-menu">
							<MediaUploadCheck>
								<MediaUpload
									title={ __( 'Select Poster Image', 'jetpack-videopress-pkg' ) }
									onSelect={ image => {
										onSelectPoster( image );
										onClose();
									} }
									allowedTypes={ VIDEO_POSTER_ALLOWED_MEDIA_TYPES }
									render={ ( { open } ) => (
										<MenuItem
											icon={ imageIcon }
											onClick={ open }
											aria-describedby={ videoPosterDescription }
										>
											{ ! poster ? selectPosterLabel : replacePosterLabel }
											<p id={ videoPosterDescription } hidden>
												{ poster
													? sprintf(
															/* translators: Placeholder is an image URL. */
															__( 'The current poster image url is %s', 'jetpack-videopress-pkg' ),
															poster
													  )
													: __(
															'There is no poster image currently selected',
															'jetpack-videopress-pkg'
													  ) }
											</p>
										</MenuItem>
									) }
								/>
							</MediaUploadCheck>

							{ !! poster && (
								<MenuItem
									onClick={ () => {
										onRemovePoster();
										onClose();
									} }
									icon={ linkOff }
								>
									{ __( 'Remove and use default', 'jetpack-videopress-pkg' ) }
								</MenuItem>
							) }
						</NavigableMenu>
						<VideoPosterCard poster={ poster } />
					</>
				);
			} }
		/>
	);
}
