<?php
namespace Mewz\WCAS\Actions\Admin\Stock;

use Mewz\Framework\Base\Action;
use Mewz\Framework\Compatibility\Multilang;
use Mewz\Framework\Util\Number;
use Mewz\WCAS\Models\AttributeStock;
use Mewz\WCAS\Util\Attributes;
use Mewz\WCAS\Util\Matches;
use Mewz\WCAS\Util\Products;

class StockList extends Action
{
	public function __hooks()
	{
		add_filter('manage_mewz_attribute_stock_posts_columns', [$this, 'columns']);
		add_filter('manage_edit-mewz_attribute_stock_sortable_columns', [$this, 'sortable_columns']);
		add_action('manage_mewz_attribute_stock_posts_custom_column', [$this, 'column_output'], 10, 2);
		add_filter('display_post_states', [$this, 'display_post_states'], 10, 2);
		add_filter('view_mode_post_types', [$this, 'view_mode_post_types']);
	}

	public function __scripts()
	{
		$this->scripts->enqueue_bundle('@admin/stock-list');

		add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts'], 100);
	}

	public function enqueue_scripts()
	{
		$this->scripts->export_js('@admin/stock-list', 'mewz_wcas_stock_list_data', [
			'rest_url' => rest_url('mewz/wcas'),
			'rest_nonce' => wp_create_nonce('wp_rest'),
			'locale' => str_replace('_', '-', get_bloginfo('language')),
		]);
	}

	public function columns($columns)
	{
		unset($columns['date']);

		$columns['actions'] = __('Actions', 'woocommerce');
		$columns['sku'] = __('SKU', 'woocommerce');
		$columns['quantity'] = __('Stock', 'woocommerce');
		$columns['notes'] = __('Notes', 'woocommerce-attribute-stock');
		$columns['attributes'] = __('Attributes', 'woocommerce');
		$columns['filters'] = __('Filters', 'woocommerce-attribute-stock');
		$columns['created'] = __('Created', 'woocommerce');
		$columns['modified'] = __('Modified', 'woocommerce-attribute-stock');

		return $columns;
	}

	public function sortable_columns($columns)
	{
		if ($all_columns = $this->columns($columns)) {
			$columns = [];

			foreach ($all_columns as $key => $label) {
				$columns[$key] = $key;
			}
		}

		unset($columns['actions']);

		return $columns;
	}

	public function column_output($column, $post_id)
	{
		global $post;

		if ($post->post_type !== 'mewz_attribute_stock') {
			return;
		}

		$stock = AttributeStock::instance($post);

		switch ($column) {
			case 'actions':
				do_action('mewz_wcas_stock_row_actions_output', $stock);
				break;

			case 'sku':
				echo esc_html($stock->sku());
				break;

			case 'quantity':
				$this->output_quantity_column($stock);
				break;

			case 'notes':
				echo esc_html($stock->notes());
				break;

			case 'attributes':
				$this->output_attributes_column($stock);
				break;

			case 'filters':
				$this->output_filters_column($stock);
				break;

			case 'created':
				echo '<abbr title="' . $stock->created(false, 'admin-full') . '">' . esc_html($stock->created(false, 'admin-date')) . '</abbr>';
				break;

			case 'modified':
				echo '<abbr title="' . $stock->modified(false, 'admin-full') . '">' . esc_html($stock->modified(false, 'admin-date')) . '</abbr>';
				break;
		}
	}

	public function output_quantity_column(AttributeStock $stock)
	{
		echo $stock->formatted_quantity();

		if ($stock->trashed()) return;

	    ?><span class="inline-edit-quantity" data-stock-id="<?= $stock->id() ?>" data-quantity="<?= Number::period_decimal($stock->quantity()) ?>"><?php

	      ?><button type="button" class="action-button edit-quantity-button" data-action="edit" title="<?= esc_attr__('Edit Stock', 'woocommerce-attribute-stock') ?>"></button><?php

	      ?><button type="button" class="action-button adjust-quantity-button" data-action="adjust" title="<?= esc_attr__('Add/Subtract Stock', 'woocommerce-attribute-stock') ?>"></button><?php

	    ?></span><?php
	}

	public function output_attributes_column(AttributeStock $stock)
	{
		$match_sets = Matches::get_sets($stock->id());
		if (!$match_sets) return;

		$attribute_list = Attributes::get_attributes();

		$attributes = [];
		$terms = [];
		$taxonomies = [];
		$all_term_ids = [];

		foreach ($match_sets as $set) {
		    foreach ($set['rows'] as $row) {
		    	$attr_id = (int)$row['attribute'];

		    	if (!isset($attribute_list[$attr_id])) {
		    		continue;
		    	}

		    	if (!isset($attributes[$attr_id])) {
				    $attributes[$attr_id] = $attribute_list[$attr_id]['label'];
					$taxonomies[$attr_id] = 'pa_' . $attribute_list[$attr_id]['name'];
		    	}

				$term_id = (int)$row['term'];

			    if (!isset($terms[$attr_id][$term_id])) {
				    $terms[$attr_id][$term_id] = '';
			    }

			    if ($term_id > 0 && !isset($all_term_ids[$term_id])) {
				    $all_term_ids[$term_id] = $term_id;
			    }
		    }
		}

		if (!$attributes) return;
		asort($attributes);

		Multilang::toggle_term_filters(false);

		$all_term_labels = $all_term_ids ? get_terms([
			'include' => array_values($all_term_ids),
			'taxonomy' => array_values($taxonomies),
			'fields' => 'id=>name',
			'orderby' => 'none',
			'hide_empty' => false,
			'update_term_meta_cache' => false,
			'suppress_filter' => true,
		]) : [];

		Multilang::toggle_term_filters(true);

		$tags = [];

		foreach ($attributes as $attr_id => $attr_label) {
			$term_labels = [];

			foreach ($terms[$attr_id] as $term_id => $_) {
				if ($term_id === 0) {
					$term_labels[$term_id] = __('Any', 'woocommerce-attribute-stock');
				} elseif (isset($all_term_labels[$term_id])) {
					$term_labels[$term_id] = $all_term_labels[$term_id];
				}
			}

			if (count($term_labels) > 1 || key($term_labels) === 0) {
				$url = 'edit-tags.php?taxonomy=pa_' . $attribute_list[$attr_id]['name'] . '&amp;post_type=product';
			} else {
				$url = 'term.php?taxonomy=pa_' . $attribute_list[$attr_id]['name'] . '&amp;tag_ID=' . key($term_labels) . '&amp;post_type=product&amp;wp_http_referer=%2Fwp-admin%2Fedit-tags.php%3Ftaxonomy%3Dpa_' . $attribute_list[$attr_id]['name'] . '%26post_type%3Dproduct';
			}

			$tag = '<a href="' . $url . '" class="mewz-wcas-label mewz-wcas-label-attribute">';

			if ($term_labels) {
				$tag .= '<span class="attribute">' . esc_html($attr_label) . ':</span>';
				$tag .= ' <span class="terms">' . implode(', ', $term_labels) . '</span>';
			} else {
				$tag .= '<span class="attribute">' . esc_html($attr_label) . '</span>';
			}

			$tags[] = $tag . '</a>';
		}

		echo implode(' ', $tags);
	}

	public function output_filters_column(AttributeStock $stock)
	{
		$labels = [
			$this->build_product_filter_label($stock->products()),
			$this->build_product_filter_label($stock->exclude_products(), 'exclude'),
			$this->build_category_filter_label($stock->categories()),
			$this->build_category_filter_label($stock->exclude_categories(), 'exclude'),
			$this->build_product_types_filter_label($stock->product_types()),
		];

		echo implode(' ', array_filter($labels));
	}

	public function build_category_filter_label($category_ids, $class = '')
	{
		$count = count($category_ids);
		if ($count === 0) return false;

		$label = '';
		$url = '';

		foreach ($category_ids as $category_id) {
			$category_label = Products::get_category_tree_label($category_id, ' > ', true);
			$count--;

			if ($category_label && !is_wp_error($category_label)) {
				$label .= $category_label;
				$url = 'term.php?taxonomy=product_cat&tag_ID=' . $category_id . '&post_type=product';
				break;
			}
		}

		if ($count) {
			$label .= ', <span class="more">+' . $count . '</span>';
		}

		if (!$label) return false;

		if (!$url) {
			$url = 'edit-tags.php?taxonomy=product_cat&post_type=product';
		}

		return '<a href="' . $url . '" class="mewz-wcas-label mewz-wcas-label-categories' . ($class ? " $class" : '') . '">' . $label . '</a>';
	}

	public function build_product_filter_label($product_ids, $class = '')
	{
		$count = count($product_ids);
		if ($count === 0) return false;

		$label = '';
		$url = '';

		foreach ($product_ids as $product_id) {
			$product = wc_get_product($product_id);
			$count--;

			if ($product) {
				$label .= Products::get_formatted_product_name($product);
				$url = get_edit_post_link($product->get_parent_id() ?: $product->get_id());
				break;
			}
		}

		if ($count) {
			$label .= ', <span class="more">+' . $count . '</span>';
		}

		if (!$label) return false;

		if (!$url) {
			$url = 'edit.php?post_type=product';
		}

		return '<a href="' . $url . '" class="mewz-wcas-label mewz-wcas-label-products' . ($class ? " $class" : '') . '">' . $label . '</a>';
	}

	public function build_product_types_filter_label($product_types)
	{
		if (!$product_types) return false;

		$product_types = array_intersect_key(Products::get_product_types(), array_flip($product_types));

		if (!$product_types) return false;

		$label = implode(', ', $product_types);
		$url = 'edit.php?post_type=product&product_type=' . implode(',', array_keys($product_types));

		return '<a href="' . $url . '" class="mewz-wcas-label mewz-wcas-label-product-types">' . $label . '</a>';
	}

	public function display_post_states($post_states, $post)
	{
		if ($post->post_type === 'mewz_attribute_stock' && isset($post_states['draft'])) {
			$post_states['draft'] = __('Disabled', 'woocommerce-attribute-stock');
		}

		return $post_states;
	}

	public function view_mode_post_types($post_types)
	{
		unset($post_types['mewz_attribute_stock']);

		return $post_types;
	}
}
