<?php
namespace Mewz\Framework\Compatibility;

class I18n
{
	public static function native_js_i18n()
	{
	    static $result;

	    if ($result === null) {
		    global $wp_version;
		    $result = version_compare($wp_version, '5.0', '>=');
	    }

	    return $result;
	}

	/**
	 * Sets a translation textdomain.
	 *
	 * @since 5.0.0
	 * @since 5.1.0 The `$domain` parameter was made optional.
	 *
	 * @param string $handle Name of the script to register a translation domain to.
	 * @param string $domain Optional. Text domain. Default 'default'.
	 * @param string $path Optional. The full file path to the directory containing translation files.
	 *
	 * @return bool True if the text domain was registered, false if not.
	 */
	public static function set_script_translations($handle, $domain = 'default', $path = null)
	{
		if (self::native_js_i18n()) {
			return wp_scripts()->set_translations($handle, $domain, $path);
		} else {
			return wp_scripts()->add_data($handle, 'mewz-translate', compact('domain', 'path'));
		}
	}

	/**
	 * Determine the current locale desired for the request.
	 *
	 * @since 5.0.0
	 *
	 * @global string $pagenow
	 *
	 * @return string The determined locale.
	 */
	public static function determine_locale()
	{
		static $locale;

		if ($locale === null) {
			if (self::native_js_i18n()) {
				$locale = determine_locale();
			} else {
				$locale = get_bloginfo('language');
			}
		}

		return $locale;
	}

	/**
	 * Loads the script translated strings.
	 *
	 * @since 5.0.0
	 * @since 5.0.2 Uses load_script_translations() to load translation data.
	 * @since 5.1.0 The `$domain` parameter was made optional.
	 *
	 * @see WP_Scripts::set_translations()
	 *
	 * @param string $handle Name of the script to register a translation domain to.
	 * @param string $domain Optional. Text domain. Default 'default'.
	 * @param string $path   Optional. The full file path to the directory containing translation files.
	 *
	 * @return false|string False if the script textdomain could not be loaded, the translated strings
	 *                      in JSON encoding otherwise.
	 */
	public static function load_script_textdomain( $handle, $domain = 'default', $path = null )
	{
		if (self::native_js_i18n()) {
			return load_script_textdomain($handle, $domain, $path);
		}

		$wp_scripts = wp_scripts();

		if ( ! isset( $wp_scripts->registered[ $handle ] ) ) {
			return false;
		}

		$path   = untrailingslashit( $path );
		$locale = self::determine_locale();

		// If a path was given and the handle file exists simply return it.
		$file_base       = $domain === 'default' ? $locale : $domain . '-' . $locale;
		$handle_filename = $file_base . '-' . $handle . '.json';

		if ( $path ) {
			$translations = self::load_script_translations( $path . '/' . $handle_filename, $handle, $domain );

			if ( $translations ) {
				return $translations;
			}
		}

		$src = $wp_scripts->registered[ $handle ]->src;

		if ( ! preg_match( '|^(https?:)?//|', $src ) && ! ( $wp_scripts->content_url && 0 === strpos( $src, $wp_scripts->content_url ) ) ) {
			$src = $wp_scripts->base_url . $src;
		}

		$relative       = false;
		$languages_path = WP_LANG_DIR;

		$src_url     = wp_parse_url( $src );
		$content_url = wp_parse_url( content_url() );
		$plugins_url = wp_parse_url( plugins_url() );
		$site_url    = wp_parse_url( site_url() );

		// If the host is the same or it's a relative URL.
		if (
			( ! isset( $content_url['path'] ) || strpos( $src_url['path'], $content_url['path'] ) === 0 ) &&
			( ! isset( $src_url['host'] ) || $src_url['host'] === $content_url['host'] )
		) {
			// Make the src relative the specific plugin or theme.
			if ( isset( $content_url['path'] ) ) {
				$relative = substr( $src_url['path'], strlen( $content_url['path'] ) );
			} else {
				$relative = $src_url['path'];
			}
			$relative = trim( $relative, '/' );
			$relative = explode( '/', $relative );

			$languages_path = WP_LANG_DIR . '/' . $relative[0];

			$relative = array_slice( $relative, 2 ); // Remove plugins/<plugin name> or themes/<theme name>.
			$relative = implode( '/', $relative );
		} elseif (
			( ! isset( $plugins_url['path'] ) || strpos( $src_url['path'], $plugins_url['path'] ) === 0 ) &&
			( ! isset( $src_url['host'] ) || $src_url['host'] === $plugins_url['host'] )
		) {
			// Make the src relative the specific plugin.
			if ( isset( $plugins_url['path'] ) ) {
				$relative = substr( $src_url['path'], strlen( $plugins_url['path'] ) );
			} else {
				$relative = $src_url['path'];
			}
			$relative = trim( $relative, '/' );
			$relative = explode( '/', $relative );

			$languages_path = WP_LANG_DIR . '/plugins';

			$relative = array_slice( $relative, 1 ); // Remove <plugin name>.
			$relative = implode( '/', $relative );
		} elseif ( ! isset( $src_url['host'] ) || $src_url['host'] === $site_url['host'] ) {
			if ( ! isset( $site_url['path'] ) ) {
				$relative = trim( $src_url['path'], '/' );
			} elseif ( ( strpos( $src_url['path'], trailingslashit( $site_url['path'] ) ) === 0 ) ) {
				// Make the src relative to the WP root.
				$relative = substr( $src_url['path'], strlen( $site_url['path'] ) );
				$relative = trim( $relative, '/' );
			}
		}

		// If the source is not from WP.
		if ( false === $relative ) {
			return self::load_script_translations( false, $handle, $domain );
		}

		// Translations are always based on the unminified filename.
		if ( substr( $relative, -7 ) === '.min.js' ) {
			$relative = substr( $relative, 0, -7 ) . '.js';
		}

		$md5_filename = $file_base . '-' . md5( $relative ) . '.json';

		if ( $path ) {
			$translations = self::load_script_translations( $path . '/' . $md5_filename, $handle, $domain );

			if ( $translations ) {
				return $translations;
			}
		}

		$translations = self::load_script_translations( $languages_path . '/' . $md5_filename, $handle, $domain );

		if ( $translations ) {
			return $translations;
		}

		return self::load_script_translations( false, $handle, $domain );
	}

	/**
	 * Loads the translation data for the given script handle and text domain.
	 *
	 * @since 5.0.2
	 *
	 * @param string|false $file   Path to the translation file to load. False if there isn't one.
	 * @param string       $handle Name of the script to register a translation domain to.
	 * @param string       $domain The text domain.
	 * @return string|false The JSON-encoded translated strings for the given script handle and text domain. False if there are none.
	 */
	public static function load_script_translations( $file, $handle, $domain )
	{
		if (self::native_js_i18n()) {
			return load_script_translations($file, $handle, $domain);
		}

		$translations = null;

		if ( null !== $translations ) {
			return $translations;
		}

		if ( ! $file || ! is_readable( $file ) ) {
			return false;
		}

		$translations = file_get_contents( $file );

		return $translations;
	}
}
