<?php
namespace Mewz\WCAS\Util;

use Mewz\Framework\Compatibility\Multilang;

class Products
{
	/** @var bool[] */
	public static $excluded = [];

	/** @var array */
	public static $categories = [];

	/** @var array */
	public static $attributes = [];

	/**
	 * Quick rudimentary check for if a product is valid for attribute stock
	 *
	 * @param \WC_Product $product
	 * @param bool $allow_variable
	 *
	 * @return bool
	 */
	public static function is_product_allowed($product, $allow_variable = false)
	{
		if (!$product instanceof \WC_Product || $product instanceof \WC_Product_Grouped || $product instanceof \WC_Product_External) {
			return false;
		}

		if (!$allow_variable && $product instanceof \WC_Product_Variable) {
			return false;
		}

	    return !self::is_product_excluded($product);
	}

	/**
	 * Checks if a product or variation has been excluded or ignored from attribute stock.
	 *
	 * @param \WC_Product|int $product Product object or ID
	 * @param \WC_Product|false $parent Provide the product's parent object if applicable
	 *                                  for improved performance, or false to skip checking parent
	 *
	 * @return bool
	 */
	public static function is_product_excluded($product, $parent = null)
	{
		if ($product instanceof \WC_Product) {
			$product_id = $product->get_id();
		} elseif ($product > 0) {
			$product_id = (int)$product;
			$product = null;
		} else {
			return false;
		}

		$check_parent = $parent === false ? 0 : 1;

		if (isset(self::$excluded[$check_parent][$product_id])) {
			return self::$excluded[$check_parent][$product_id];
		}

		if ($product) {
			$excluded = (bool)$product->get_meta('_mewz_wcas_exclude');
		} else {
			$excluded = (bool)get_post_meta($product_id, '_mewz_wcas_exclude', true);
		}

		if (!$excluded && $check_parent) {
			if (!$parent) {
				$parent = $product ? $product->get_parent_id() : wp_get_post_parent_id($product_id);
			}

			if ($parent) {
				$parent_id = $parent instanceof \WC_Product ? $parent->get_id() : (int)$parent;

				if ($parent_id !== $product_id) {
					$excluded = self::is_product_excluded($parent);
				}
			}
		}

		$excluded = apply_filters('mewz_wcas_is_product_excluded', $excluded, $product_id);

		return self::$excluded[$check_parent][$product_id] = $excluded;
	}

	/**
	 * Gets a complete list of product attribute data.
	 *
	 * @param \WC_Product $product
	 * @param array $variation Variation attribute data to merge
	 * @param bool $assoc Key by attribute name instead of ID for non-variations
	 *
	 * @return array
	 */
	public static function get_product_attributes(\WC_Product $product, $variation = null, $assoc = false)
	{
		$attributes = [];

		if ($parent_id = $product->get_parent_id()) {
			// temporarily cache parent attributes to avoid excessive `wc_get_product()` calls
			if (isset(self::$attributes[$parent_id])) {
				$attributes = self::$attributes[$parent_id];
			}
			elseif ($parent = wc_get_product($parent_id)) {
				// get non-variation attributes from parent product
				/** @var \WC_Product_Attribute $attr */
				foreach ($parent->get_attributes() as $attr) {
					if ($attr->get_id() && !$attr->get_variation() && $terms = $attr->get_options()) {
						$attributes[$attr->get_name()] = $terms;
					}
				}

				self::$attributes[$parent_id] = $attributes;
			}
		}

		foreach ($product->get_attributes() as $key => $attr) {
		    if ($attr instanceof \WC_Product_Attribute) {
			    $attr_id = $attr->get_id();

			    if ($attr_id && $terms = $attr->get_options()) {
				    $key = $assoc ? $attr->get_name() : $attr_id;
				    $attributes[$key] = $terms;
			    }
		    } else {
			    $attributes[wc_sanitize_taxonomy_name($key)] = $attr;
		    }
		}

		// get variation attribute data attached to product object
		if ($variation === null && property_exists($product, 'mewz_wcas_variation') && $product->mewz_wcas_variation) {
			$variation = $product->mewz_wcas_variation;
		}

		// merge additional variation attribute data
		if (is_array($variation) && $variation) {
			$variation_data = Attributes::strip_attribute_prefix($variation);
			$variation_data = Attributes::decode_keys($variation_data);

			$attributes = $variation_data + $attributes;
		}

		return apply_filters('mewz_wcas_product_attributes', $attributes, $product, $variation, $assoc);
	}

	/**
	 * Gets a complete list of product category IDs (including ancestors).
	 *
	 * @param \WC_Product|int $product Checks for product parent only if object passed
	 * @param bool $bypass_multilang
	 *
	 * @return array
	 */
	public static function get_all_product_category_ids($product, $bypass_multilang = true)
	{
		if ($product instanceof \WC_Product) {
			$product_id = $product->get_parent_id() ?: $product->get_id();
		} else {
			$product_id = (int)$product;
		}

		if (!isset(self::$categories[$product_id])) {
			if ($bypass_multilang) {
				Multilang::toggle_term_filters(false);
			}

			$categories = wc_get_product_term_ids($product_id, 'product_cat');

			if ($categories) {
				$ancestors = [];

				foreach ($categories as $cat_id) {
					$ancestors[] = get_ancestors($cat_id, 'product_cat', 'taxonomy');
				}

				self::$categories[$product_id] = array_merge($categories, ...$ancestors);
			} else {
				self::$categories[$product_id] = [];
			}

			if ($bypass_multilang) {
				Multilang::toggle_term_filters(true);
			}
		}

		return self::$categories[$product_id];
	}

	/**
	 * @param \WP_Term|int $category
	 *
	 * @return \WP_Term[]|\WP_Error|null
	 */
	public static function get_category_ancestry($category)
	{
		if (!$category instanceof \WP_Term) {
			$category = get_term($category, 'product_cat');
		}

		if (!$category || is_wp_error($category)) {
			return $category;
		}

		$tree[] = $category;

		while ($category->parent > 0 && ($category = get_term($category->parent, 'product_cat')) && !is_wp_error($category)) {
			$tree[] = $category;
		}

		return $tree;
	}

	/**
	 * @param \WP_Term|int $category
	 * @param string $sep
	 * @param bool|string $compress
	 *
	 * @return string|\WP_Error|null
	 */
	public static function get_category_tree_label($category, $sep = ' > ', $compress = false)
	{
		$ancestry = self::get_category_ancestry($category);

		if (!$ancestry || is_wp_error($ancestry)) {
			return $ancestry;
		}

		$count = count($ancestry);

		if ($count === 1) {
			$label = $ancestry[0]->name;
		} elseif ($count === 2) {
			$label = $ancestry[1]->name . $sep . $ancestry[0]->name;
		} elseif ($compress) {
			if (is_string($compress)) {
				$sep = $compress;
			}

			$label = end($ancestry)->name . $sep . $ancestry[0]->name;
		} else {
			$label = [];

			foreach (array_reverse($ancestry) as $category) {
				$label[] = $category->name;
			}

			$label = implode($sep, $label);
		}

		return $label;
	}

	/**
	 * @return string[]
	 */
	public static function get_product_types()
	{
		static $product_types;

		if ($product_types === null) {
			$product_types = wc_get_product_types();

			unset($product_types['grouped'], $product_types['external']);

			// remove "product" from product type label (at least for English)
			foreach ($product_types as &$product_type) {
				$product_type = trim(str_replace('product', '', $product_type));
			}
		}

		return $product_types;
	}

	/**
	 * @param \WC_Product $product
	 *
	 * @return string
	 */
	public static function get_formatted_product_name($product)
	{
		$name = $product->get_name();

		if (!$product instanceof \WC_Product_Variation) {
			return $name;
		}

		$sku = $product->get_sku('edit');

		if ($sku !== '') {
			$name .= " ($sku)";
		} else {
			/** @var \WC_Product_Variation $product */
			$variation = wc_get_formatted_variation($product, true, true, true);

			$name .= $variation !== '' ? " ($variation)" : ' (#' . $product->get_id() . ')';
		}

		return $name;
	}
}
