<?php
namespace Mewz\WCAS\Actions\Front;

use Mewz\Framework\Base\Action;
use Mewz\Framework\Util\WooCommerce;
use Mewz\WCAS\Util\Limits;
use Mewz\WCAS\Util\Products;
use Mewz\WCAS\Util\Settings;

class ProductLimits extends Action
{
	public static $enabled = true;
	public static $limit_variable = false;

	public function __hooks()
	{
		add_filter('woocommerce_product_get_manage_stock', [$this, 'get_manage_stock'], 5, 2);
		add_filter('woocommerce_product_get_stock_quantity', [$this, 'get_stock_quantity'], 5, 2);
		add_filter('woocommerce_product_get_stock_status', [$this, 'get_stock_status'], 5, 2);

		add_filter('woocommerce_product_variation_get_manage_stock', [$this, 'get_manage_stock'], 5, 2);
		add_filter('woocommerce_product_variation_get_stock_quantity', [$this, 'get_stock_quantity'], 5, 2);
		add_filter('woocommerce_product_variation_get_stock_status', [$this, 'get_stock_status'], 5, 2);
		add_filter('woocommerce_product_variation_get_backorders', [$this, 'variation_get_backorders'], 5, 2);
		add_filter('woocommerce_get_children', [$this, 'get_children'], 5, 3);

		add_action('woocommerce_before_product_object_save', [$this, 'before_product_object_save'], 5);
		add_filter('woocommerce_order_item_product', [$this, 'order_item_product'], 0, 2);

		if (Settings::allow_backorders() !== 'no') {
			add_filter('woocommerce_product_get_backorders', [$this, 'allow_backorders'], 6, 2);
			add_filter('woocommerce_product_variation_get_backorders', [$this, 'allow_backorders'], 6, 2);
		}
	}

	public static function allow_product(\WC_Product $product, $allow_variable = false)
	{
		return (!property_exists($product, 'mewz_wcas_bypass_limits') || !$product->mewz_wcas_bypass_limits) && Products::is_product_allowed($product, $allow_variable || self::$limit_variable);
	}

	public static function allow_stock_status(\WC_Product $product)
	{
		return apply_filters('mewz_wcas_limit_stock_status', $product->get_stock_status('edit') === 'instock', $product);
	}

	public function get_manage_stock($value, \WC_Product $product)
	{
		if ($value || !self::$enabled || !self::allow_product($product)) {
			return $value;
		}

		if (self::allow_stock_status($product) && Limits::get_stock_limit($product)) {
			$value = true;

			// save original 'stock_status' before `$product->validate_props()` changes it in 'view' context (with filters) before saving!!
			$product->mewz_wcas_original_stock_status = $product->get_stock_status('edit');
		}

		return $value;
	}

	public function get_stock_quantity($value, \WC_Product $product)
	{
		if (!self::$enabled || !self::allow_product($product)) {
			return $value;
		}

		if (!apply_filters('mewz_wcas_limit_product_stock_quantity', true, $product, $value)) {
			return $value;
		}

		$product->mewz_wcas_bypass_limits = true;

		$manage_stock = $product->get_manage_stock();

		if (($manage_stock || self::allow_stock_status($product)) && $limit = Limits::get_stock_limit($product)) {
			if ($product instanceof \WC_Product_Variation && $manage_stock === 'parent' && $parent_data = $product->get_parent_data()) {
				$value = apply_filters('woocommerce_product_variation_get_stock_quantity', $parent_data['stock_quantity'], $product);
			}

			if (!$manage_stock || $limit['limit_qty'] < $value) {
				$product->mewz_wcas_original_quantity = $value;
				$value = $limit['limit_qty'];
			}
		}

		$product->mewz_wcas_bypass_limits = false;

		return $value;
	}

	public function get_stock_status($value, \WC_Product $product)
	{
		if (!$value || !self::$enabled || !self::allow_product($product, true)) {
			return $value;
		}

		$product->mewz_wcas_bypass_limits = true;
		$override = ($product->get_manage_stock() || self::allow_stock_status($product)) && !$product->backorders_allowed();
		$product->mewz_wcas_bypass_limits = false;

		if (!$override) {
			return $value;
		}

		if ($product instanceof \WC_Product_Variable) {
			return $this->variable_get_stock_status($value, $product);
		}

		$limit = Limits::get_stock_limit($product);

		if ($limit && $limit['limit_qty'] <= WooCommerce::no_stock_amount()) {
			$value = 'outofstock';
		}

		return $value;
	}

	protected function variable_get_stock_status($value, \WC_Product $product)
	{
		$cache_key = 'children_stock_status';
		$cache_tags = ['match_sets', 'stock', 'product_' . $product->get_id()];

		$stock_status = $this->cache->get($cache_key, $cache_tags);

		if ($stock_status === null) {
			$stock_status = 'outofstock';

			foreach ($product->get_children() as $variation_id) {
				if (($variation = wc_get_product($variation_id)) && $variation->is_in_stock()) {
					$stock_status = $value;
					break;
				}
			}

			$this->cache->set($cache_key, $stock_status, $cache_tags);
		}

		return $stock_status;
	}

	public function variation_get_backorders($value, \WC_Product_Variation $variation)
	{
		if (!self::$enabled || (property_exists($variation, 'mewz_wcas_bypass_limits') && $variation->mewz_wcas_bypass_limits) || Products::is_product_excluded($variation)) {
			return $value;
		}

		$variation->mewz_wcas_bypass_limits = true;

		if ($variation->get_manage_stock() === 'parent' && $parent_data = $variation->get_parent_data()) {
			$value = apply_filters('woocommerce_product_variation_get_backorders', $parent_data['backorders'], $variation);
		}

		$variation->mewz_wcas_bypass_limits = false;

		return $value;
	}

	public function get_children($variation_ids, $product, $visible_only = false)
	{
		if (!$visible_only || !self::$enabled || !WooCommerce::hide_out_of_stock() || !self::allow_product($product, true)) {
			return $variation_ids;
		}

		$cache_key = 'visible_children';
		$cache_tags = ['match_sets', 'stock', 'product_' . $product->get_id()];

		$visible_ids = $this->cache->get($cache_key, $cache_tags);

		if ($visible_ids === null) {
			_prime_post_caches($variation_ids);
			$visible_ids = [];

			foreach ($variation_ids as $variation_id) {
				if (($variation = wc_get_product($variation_id)) && $variation->is_in_stock()) {
					$visible_ids[] = $variation_id;
				}
			}

			$this->cache->set($cache_key, $visible_ids, $cache_tags);
		}

		return $visible_ids;
	}

	public function before_product_object_save(\WC_Product $product)
	{
		if (!property_exists($product, 'mewz_wcas_original_stock_status') || !$product->mewz_wcas_original_stock_status || !self::allow_product($product)) {
			return;
		}

		// set 'stock_status' back to original before `$product->validate_props()` changed it incorrectly (with 'view' context)
		$product->set_stock_status($product->mewz_wcas_original_stock_status);

		// update 'stock_status' properly this time (without stock limit filters applied!)
		$product->mewz_wcas_bypass_limits = true;
		$product->validate_props();
		$product->mewz_wcas_bypass_limits = false;
	}

	public function order_item_product($product, $order_item)
	{
		/**
		 * Don't limit order item product stock else stock reduction calculations will be thrown out.
		 *
		 * But we still need to limit stock when the order item is being created, so that things like
		 * backorder meta can be added accordingly.
		 */
		if ($product && $order_item && $order_item->get_id()) {
			$product->mewz_wcas_bypass_limits = true;
		}

		return $product;
	}

	public function allow_backorders($value, \WC_Product $product)
	{
		if ($value !== 'no' || !self::$enabled || !Products::is_product_allowed($product, true)) {
			return $value;
		}

		if (
			!$product->get_manage_stock('edit')
			&& self::allow_stock_status($product)
			&& (
				(property_exists($product, 'mewz_wcas_allow_backorders_override') && $product->mewz_wcas_allow_backorders_override)
				|| Limits::get_stock_limit($product)
			)
	) {
			$value = Settings::allow_backorders(true);
		}

		return $value;
	}
}
