<?php

defined('ABSPATH') || die('Restricted Access');

class LpcLabelOutwardDeleteAction extends LpcComponent {
	const AJAX_TASK_NAME           = 'label/outward/delete';
	const TRACKING_NUMBER_VAR_NAME = 'lpc_label_tracking_number';

	/** @var LpcAjax */
	protected $ajaxDispatcher;
	/** @var LpcOutwardLabelDb */
	protected $outwardLabelDb;
	/** @var LpcAdminNotices */
	protected $adminNotices;
	/** @var LpcInwardLabelDb */
	protected $inwardLabelDb;

	public function __construct(
		LpcAjax $ajaxDispatcher = null,
		LpcOutwardLabelDb $outwardLabelDb = null,
		LpcInwardLabelDb $inwardLabelDb = null,
		LpcAdminNotices $adminNotices = null
	) {
		$this->ajaxDispatcher = LpcRegister::get('ajaxDispatcher', $ajaxDispatcher);
		$this->outwardLabelDb = LpcRegister::get('outwardLabelDb', $outwardLabelDb);
		$this->inwardLabelDb  = LpcRegister::get('inwardLabelDb', $inwardLabelDb);
		$this->adminNotices   = LpcRegister::get('lpcAdminNotices', $adminNotices);
	}

	public function getDependencies() {
		return ['ajaxDispatcher', 'outwardLabelDb', 'lpcAdminNotices'];
	}

	public function init() {
		$this->listenToAjaxAction();
	}

	protected function listenToAjaxAction() {
		$this->ajaxDispatcher->register(self::AJAX_TASK_NAME, array($this, 'control'));
	}

	public function getUrlForTrackingNumber($trackingNumber) {
		return $this->ajaxDispatcher->getUrlForTask(self::AJAX_TASK_NAME) . '&' . self::TRACKING_NUMBER_VAR_NAME . '=' . $trackingNumber;
	}

	public function control() {
		if (!current_user_can('edit_posts')) {
			header('HTTP/1.0 401 Unauthorized');

			return $this->ajaxDispatcher->makeAndLogError(
				array(
					'message' => 'unauthorized access to outward label deletion',
				)
			);
		}

		$trackingNumber      = LpcHelper::getVar(self::TRACKING_NUMBER_VAR_NAME);
		$inwardLabelsRelated = $this->inwardLabelDb->getLabelsInfosForOutward($trackingNumber);

		LpcLogger::debug(
			'Delete outward label',
			array(
				'tracking_number'       => $trackingNumber,
				'related_inward_labels' => $inwardLabelsRelated,
				'method'                => __METHOD__,
			)
		);

		$result = $this->outwardLabelDb->delete($trackingNumber);

		if (1 != $result) {
			LpcLogger::error(
				'Unable to delete label',
				array(
					'tracking_number' => $trackingNumber,
					'result'          => $result,
					'method'          => __METHOD__,
				)
			);

			$this->adminNotices->add_notice(
				'outward_label_delete',
				'notice-error',
				sprintf(
					__('Unable to delete label %s', 'wc_colissimo'),
					$trackingNumber
				)
			);
		} else {
			$noticeText = sprintf(__('Label %s deleted', 'wc_colissimo'), $trackingNumber);
			if (count($inwardLabelsRelated) > 0) {
				$inwardDeletionResult = $this->inwardLabelDb->deleteForOutward($trackingNumber);

				if (count($inwardLabelsRelated) != $inwardDeletionResult) {
					LpcLogger::error(
						'Unable to delete some inwards label related to ouwtard',
						array(
							'tracking_number'       => $trackingNumber,
							'related_inward_labels' => $inwardLabelsRelated,
							'result'                => $inwardDeletionResult,
							'method'                => __METHOD__,
						)
					);

				} else {
					foreach ($inwardLabelsRelated as $oneInwardLabel) {
						$noticeText .= '<br>' . sprintf(
								__('Inward label %s deleted', 'wc_colissimo'),
								$oneInwardLabel->tracking_number
							);
					}
				}
			}

			$this->adminNotices->add_notice(
				'outward_label_delete',
				'notice-success',
				$noticeText
			);
		}

		wp_redirect(admin_url('admin.php?page=wc_colissimo_view'));
	}
}
