<?php

defined('ABSPATH') || die('Restricted Access');

class LpcLabelInwardDeleteAction extends LpcComponent {
	const AJAX_TASK_NAME           = 'label/inward/delete';
	const TRACKING_NUMBER_VAR_NAME = 'lpc_label_tracking_number';

	/** @var LpcAjax */
	protected $ajaxDispatcher;
	/** @var LpcInwardLabelDb */
	protected $inwardLabelDb;
	/** @var LpcAdminNotices */
	protected $adminNotices;

	public function __construct(
		LpcAjax $ajaxDispatcher = null,
		LpcInwardLabelDb $inwardLabelDb = null,
		LpcAdminNotices $adminNotices = null
	) {
		$this->ajaxDispatcher = LpcRegister::get('ajaxDispatcher', $ajaxDispatcher);
		$this->inwardLabelDb  = LpcRegister::get('inwardLabelDb', $inwardLabelDb);
		$this->adminNotices   = LpcRegister::get('lpcAdminNotices', $adminNotices);
	}

	public function getDependencies() {
		return ['ajaxDispatcher', 'inwardLabelDb', 'lpcAdminNotices'];
	}

	public function init() {
		$this->listenToAjaxAction();
	}

	protected function listenToAjaxAction() {
		$this->ajaxDispatcher->register(self::AJAX_TASK_NAME, array($this, 'control'));
	}

	public function getUrlForTrackingNumber($trackingNumber) {
		return $this->ajaxDispatcher->getUrlForTask(self::AJAX_TASK_NAME) . '&' . self::TRACKING_NUMBER_VAR_NAME . '=' . $trackingNumber;
	}

	public function control() {
		if (!current_user_can('edit_posts')) {
			header('HTTP/1.0 401 Unauthorized');

			return $this->ajaxDispatcher->makeAndLogError(
				array(
					'message' => 'unauthorized access to outward label deletion',
				)
			);
		}

		$trackingNumber = LpcHelper::getVar(self::TRACKING_NUMBER_VAR_NAME);

		LpcLogger::debug(
			'Delete inward label',
			array(
				'tracking_number' => $trackingNumber,
				'method'          => __METHOD__,
			)
		);

		$result = $this->inwardLabelDb->delete($trackingNumber);

		if (1 != $result) {
			LpcLogger::error(
				'Unable to delete label',
				array(
					'tracking_number' => $trackingNumber,
					'result'          => $result,
					'method'          => __METHOD__,
				)
			);

			$this->adminNotices->add_notice('inward_label_delete', 'notice-error',
				sprintf(__('Unable to delete label %s', 'wc_colissimo'), $trackingNumber));
		} else {
			$this->adminNotices->add_notice('inward_label_delete', 'notice-success',
				sprintf(__('Label %s deleted', 'wc_colissimo'), $trackingNumber));
		}

		wp_redirect(admin_url('admin.php?page=wc_colissimo_view'));
	}
}
